from datetime import datetime
from typing import List, Optional, Tuple

from sendr_interactions.clients.balance.base import BaseBalanceClient
from sendr_interactions.clients.balance.entities import CollateralType, OfferConfirmationType, PaymentType
from sendr_utils import without_none


class BalanceOfferClient(BaseBalanceClient):
    async def create_offer(
        self,
        uid: int,
        client_id: str,
        person_id: str,
        services: List[int],
        payment_type: PaymentType,
        firm_id: int,
        currency: str,
        payment_term_days: int,
        manager_code: int,
        partner_credit: Optional[int] = None,
        offer_confirmation_type: Optional[OfferConfirmationType] = None,
        partner_commission_percentage: Optional[float] = None,
        integration_cc: Optional[str] = None,
        extra_params: Optional[dict] = None,
    ) -> Tuple[int, str]:
        """
        Creates offer in balance.
        Returns:
            ID, EXTERNAL_ID pair for created Offer
        """
        assert payment_term_days > 0

        offer_data = without_none({
            'client_id': client_id,
            'person_id': person_id,
            'services': services,
            'payment_type': payment_type.value,
            'partner_credit': partner_credit,
            'firm_id': firm_id,
            'currency': currency,
            'payment_term': payment_term_days,
            'offer_confirmation_type': offer_confirmation_type.value if offer_confirmation_type is not None else None,
            'partner_commission_pct2': partner_commission_percentage,
            'manager_code': manager_code,
            'integration': integration_cc,
            **(extra_params or {})
        })
        result = await self.request(
            'create_offer',
            method_name='CreateOffer',
            data=(str(uid), offer_data),
        )
        contract_data, = result
        return contract_data['ID'], str(contract_data['EXTERNAL_ID'])

    async def create_collateral(self, uid: int, contract_id: str, collateral_type: CollateralType) -> int:
        """
        Creates collateral for existing contract. Returns it's id.
        """
        collateral_data = {'END_DT': datetime.now()}
        result = await self.request(
            'create_collateral',
            method_name='CreateCollateral',
            data=(str(uid), contract_id, collateral_type.value, collateral_data),
        )
        contract_data, = result
        return contract_data['CONTRACT_ID']
