from datetime import datetime
from typing import Any, Iterable, Optional

import pytz

from sendr_interactions.clients.balance.base import BaseBalanceClient
from sendr_interactions.clients.balance.entities import Person
from sendr_utils import without_none


class BalancePersonClient(BaseBalanceClient):
    @staticmethod
    def _format_address(postcode: Optional[str],
                        city: Optional[str],
                        street: Optional[str],
                        home: Optional[str],
                        ) -> str:
        if postcode is None or city is None or street is None or home is None:
            return ''
        return ', '.join((postcode, city, street, home))

    @staticmethod
    def _init_person(data: dict, is_safe: bool = False) -> Person:
        def get_field(field: str) -> str:
            return data.get(field, '') if is_safe else data[field]

        return Person(
            client_id=get_field('CLIENT_ID'),
            person_id=data.get('ID') if is_safe else data['ID'],

            account=data.get('ACCOUNT'),
            bik=data.get('BIK'),
            fname=get_field('FNAME'),
            lname=get_field('LNAME'),
            mname=data.get('MNAME'),
            email=get_field('EMAIL'),
            phone=get_field('PHONE'),

            name=get_field('NAME'),
            longname=get_field('LONGNAME'),
            inn=get_field('INN'),
            kpp=data.get('KPP'),
            ogrn=data.get('OGRN'),

            legal_address_city=get_field('LEGAL_ADDRESS_CITY'),
            legal_address_home=get_field('LEGAL_ADDRESS_HOME'),
            legal_address_postcode=get_field('LEGAL_ADDRESS_POSTCODE'),
            legal_address_street=get_field('LEGAL_ADDRESS_STREET'),

            address_city=data.get('ADDRESS_CITY'),
            address_home=data.get('ADDRESS_HOME'),
            address_postcode=data.get('ADDRESS_POSTCODE'),
            address_street=data.get('ADDRESS_STREET'),

            date=(
                datetime.strptime(data['DT'], '%Y-%m-%d %H:%M:%S').replace(tzinfo=pytz.timezone('Europe/Moscow'))
                if 'DT' in data
                else None
            )
        )

    async def create_person(self, uid: int, person: Person) -> str:
        """
        Creates or updates person. Returns id.
        """
        assert person.has_legal_address
        assert person.has_postcode
        legaladdress = person.legal_address or self._format_address(
            person.legal_address_postcode,
            person.legal_address_city,
            person.legal_address_street,
            person.legal_address_home,
        )
        postaddress = person.post_address or self._format_address(
            person.address_postcode,
            person.address_city,
            person.address_street,
            person.address_home,
        ) or legaladdress
        person_data = {
            'client_id': person.client_id,
            'type': 'ur',

            'account': person.account,
            'bik': person.bik,
            'fname': person.fname,
            'lname': person.lname,
            'mname': person.mname or '',  # действительно ли тут нужна пустая строка?
            'email': person.email,
            'phone': person.phone,

            'name': person.name,
            'longname': person.longname,
            'inn': person.inn,
            'ogrn': person.ogrn,

            'legal_address_city': person.legal_address_city,
            'legal_address_home': person.legal_address_home,
            'legal_address_postcode': person.legal_address_postcode,
            'legal_address_street': person.legal_address_street,

            'address_city': person.address_city,
            'address_home': person.address_home,
            'address_postcode': person.address_postcode,
            'address_street': person.address_street,

            'postcode': person.get_postcode(),
            'postaddress': postaddress,
            'legaladdress': legaladdress,

            'kpp': person.kpp,
            'person_id': person.person_id,
        }
        result = await self.request(
            'create_person',
            method_name='CreatePerson',
            data=(str(uid), without_none(person_data)),
        )
        person_id, = result
        return str(person_id)

    async def _get_client_persons(self, client_id: str, **kwargs: Any) -> Iterable[dict]:
        result = await self.request(
            'get_client_persons',
            method_name='GetClientPersons',
            data=(client_id,),
            **kwargs
        )
        return result[0]

    async def _get_person(self, person_id: str) -> dict:
        result = await self.request(
            'get_person',
            method_name='GetPerson',
            data=({"ID": person_id},)
        )
        return result[0][0]

    async def get_client_persons(self, client_id: str, total_timeout: Optional[float] = None) -> Iterable[Person]:
        """
        Returns list of found persons.
        If client does not exist, raises BalanceClientNotFoundError.
        """
        return [
            self._init_person(person_data, is_safe=True)
            for person_data in await self._get_client_persons(
                client_id,
                **self._get_timeout_kwargs(total_timeout)
            )
        ]

    async def get_person(self, person_id: str) -> Optional[Person]:
        person_data = await self._get_person(person_id)
        return None if person_data is None else self._init_person(person_data)
