from typing import Any, Dict, List, Optional

from aiohttp import ClientResponse

from sendr_interactions.base import AbstractInteractionClient, ResponseFormat
from sendr_interactions.clients.blackbox.entities import EmailsMode, OauthResult, SessionIdResult, UserInfo
from sendr_interactions.clients.blackbox.exceptions import (
    BlackBoxInvalidSessionError, UnknownBlackboxError, UserNotFoundBlackboxError
)
from sendr_interactions.clients.blackbox.schemas import UserInfoSchema
from sendr_utils import MISSING, MaybeMissing, SentinelMissing, without_none


class AbstractBlackBoxClient(AbstractInteractionClient[ClientResponse]):
    SERVICE = 'blackbox'
    RESPONSE_FORMAT = ResponseFormat.ORIGINAL
    VALID_STATUSES = ('VALID', 'NEED_RESET')

    def _check_is_auth_valid(self, method: str, response: Dict[str, Any]) -> None:
        status = response.get('status', {}).get('value')
        error = response.get('error')
        if status not in self.VALID_STATUSES:
            raise BlackBoxInvalidSessionError(
                method=method,
                params={
                    'status': status,
                    'error': error,
                }
            )

    async def get_session_info(
        self,
        host: str,
        session_id: str,
        user_ip: str,
        default_uid: Optional[int] = None,
        get_user_ticket: bool = False,
    ) -> SessionIdResult:
        params = without_none({
            'method': 'sessionid',
            'userip': user_ip,
            'host': host,
            'sessionid': session_id,
            'default_uid': default_uid,
            'get_user_ticket': 'yes' if get_user_ticket else None,
            'get_login_id': 'yes',
            'format': 'json',
            'aliases': '13',
        })
        resp = await self.get(
            interaction_method='get_session_info',
            url=self.BASE_URL,
            params=params,
            response_log=False,
        )
        response = await resp.json()

        self._check_is_auth_valid('get_session_info', response)

        uid = int(response['uid']['value'])
        login_id = response['login_id']
        tvm_ticket = response.get('user_ticket') if get_user_ticket else None
        is_yandexoid = '13' in response.get('aliases', {})
        return SessionIdResult(uid, login_id, tvm_ticket, is_yandexoid)

    async def get_oauth_token_info(
        self,
        oauth_token: str,
        user_ip: str,
        scopes: Optional[List[str]] = None,
        get_user_ticket: bool = False,
    ) -> OauthResult:
        resp = await self.get(
            interaction_method='get_oauth_token_info',
            url=self.BASE_URL,
            params=without_none({
                'method': 'oauth',
                'userip': user_ip,
                'format': 'json',
                'oauth_token': oauth_token,
                'scopes': ','.join(scopes) if scopes else None,
                'get_user_ticket': 'yes' if get_user_ticket else None,
                'get_login_id': 'yes',
                'aliases': '13',
            }),
            response_log=False,
        )
        response = await resp.json()

        self._check_is_auth_valid('get_oauth_token_info', response)

        uid = int(response['oauth']['uid']) if response['oauth']['uid'] else None
        client_id = response['oauth']['client_id']
        login_id = response.get('login_id')
        tvm_ticket = response.get('user_ticket') if get_user_ticket else None
        is_yandexoid = '13' in response.get('aliases', {})
        return OauthResult(client_id, uid, login_id, tvm_ticket, is_yandexoid)

    async def get_user_info(
        self,
        uid: int,
        user_ip: str,
        attributes: MaybeMissing[List[str]] = MISSING,
        emails_mode: EmailsMode = EmailsMode.NONE,
    ) -> UserInfo:
        user_id = str(uid)
        params = without_none({
            'method': 'userinfo',
            'uid': user_id,
            'userip': user_ip,
            'format': 'json',
        })
        if not isinstance(attributes, SentinelMissing) and len(attributes) > 0:
            params['attributes'] = ','.join(attributes)
        if emails_mode != EmailsMode.NONE:
            params['emails'] = emails_mode.value

        response = await self.get(
            interaction_method='get_user_info',
            url=self.BASE_URL,
            params=params,
            response_log=False,
        )
        response_data = await response.json()

        try:
            user_data = next(user for user in response_data.get('users', []) if user['id'] == user_id)
        except StopIteration:
            raise UnknownBlackboxError(status_code=response.status, method='get_user_info')

        user = UserInfoSchema().load_one(user_data)
        if user.uid_data.value != uid:
            raise UserNotFoundBlackboxError(status_code=response.status, method='get_user_info')

        return user

    async def have_plus(self, uid: int, user_ip: str) -> bool:
        user_info = await self.get_user_info(uid=uid, user_ip=user_ip, attributes=['1015'])
        return '1015' in user_info.attributes
