from dataclasses import dataclass, field
from enum import Enum, unique
from typing import Dict, List, Optional, Union


@dataclass
class SessionIdResult:
    uid: int
    login_id: str
    tvm_ticket: Optional[str] = field(init=True, repr=False, default=None)
    is_yandexoid: bool = False


@dataclass
class OauthResult:
    client_id: str
    uid: Optional[int] = None
    login_id: Optional[str] = None
    tvm_ticket: Optional[str] = field(init=True, repr=False, default=None)
    is_yandexoid: bool = False


BlackBoxResult = Union[SessionIdResult, OauthResult]


@dataclass
class UIDData:
    value: Optional[int] = None


@dataclass
class Email:
    address: str
    default: bool
    native: bool


@dataclass
class Emails:
    address_list: List[Email]

    @property
    def default(self) -> Optional[Email]:
        for email in self.address_list:
            if email.default:
                return email
        return None


@dataclass
class UserInfo:
    uid_data: UIDData
    address_list: Optional[List[Email]] = None
    attributes: Dict[str, str] = field(default_factory=dict)

    @property
    def emails(self) -> Emails:
        assert self.address_list is not None
        return Emails(address_list=self.address_list)

    @property
    def uid(self) -> int:
        assert self.uid_data.value is not None
        return self.uid_data.value


@unique
class EmailsMode(Enum):
    """https://docs.yandex-team.ru/blackbox/concepts/emails"""
    NONE = 'none'
    ALL = 'getall'
    DEFAULT = 'getdefault'
