from dataclasses import dataclass
from datetime import date, datetime
from enum import Enum, unique
from typing import ClassVar, Dict, List, Optional, Union


def pick_opt_str(value: Optional[str], default: Optional[str]) -> Optional[str]:
    """ Выбирает непустое значение, если это возможно. Сохраняет пустое значение, если default не определен
    """
    return (
        default
        if value is None or (not value and default is not None)
        else value
    )


@unique
class MerchantType(Enum):
    OOO = 'ooo'
    IP = 'ip'


@unique
class PersonType(Enum):
    CEO = 'ceo'
    CONTACT = 'contact'
    SIGNER = 'signer'


@dataclass
class AddressData:
    type: str
    city: str
    country: str
    street: str
    zip: str
    home: Optional[str] = None


@dataclass
class PersonData:
    type: PersonType
    name: str
    email: str
    phone: str
    surname: str
    patronymic: Optional[str] = None
    birth_date: Optional[date] = None


@dataclass
class OrganizationData:
    type: Optional[MerchantType] = None
    name: Optional[str] = None
    english_name: Optional[str] = None
    full_name: Optional[str] = None
    inn: Optional[str] = None
    kpp: Optional[str] = None
    ogrn: Optional[str] = None
    schedule_text: Optional[str] = None
    site_url: Optional[str] = None
    description: Optional[str] = None


@unique
class SparkMethod(Enum):
    AUTH = 'Authmethod'
    END = 'End'
    GET_COMPANY = 'GetCompanyShortReport'
    GET_ENTREPRENEUR = 'GetEntrepreneurShortReport'


@unique
class XmlNs(Enum):
    SOAP = 'http://schemas.xmlsoap.org/soap/envelope/'
    IFAX = 'http://interfax.ru/ifax'

    ALIAS_SOAP = 'soap'
    ALIAS_IFAX = 'ifax'

    _ignore_ = ['NAMESPACES']
    NAMESPACES: ClassVar[Dict[str, str]]


XmlNs.NAMESPACES = {
    XmlNs.ALIAS_SOAP.value: XmlNs.SOAP.value,
    XmlNs.ALIAS_IFAX.value: XmlNs.IFAX.value,
}


@unique
class SoapTag(Enum):
    ENVELOPE = 'Envelope'
    BODY = 'Body'


@dataclass
class SparkOrganizationData:
    organization: OrganizationData
    actual_date: Optional[date]


@dataclass
class OkvedItem:
    main: bool
    code: str
    name: str


@dataclass
class LeaderData:
    name: str
    position: Optional[str] = None
    inn: Optional[str] = None
    birth_date: Optional[date] = None
    actual_date: Optional[date] = None


@dataclass
class SparkAddressData:
    address: AddressData
    actual_date: Optional[date] = None


@dataclass
class PhoneData:
    code: str
    number: str
    verification_date: Optional[date] = None


@dataclass
class SparkData:
    spark_id: str
    organization_data: SparkOrganizationData
    okved_list: Optional[List[OkvedItem]] = None
    registration_date: Optional[date] = None
    leaders: Optional[List[LeaderData]] = None
    addresses: Optional[List[SparkAddressData]] = None
    phones: Optional[List[PhoneData]] = None
    active: Optional[bool] = None

    @property
    def phone(self) -> Optional[PhoneData]:
        phones = filter(lambda phone: phone.code and phone.number, self.phones or [])
        return next(phones, None)

    @property
    def leader(self) -> Optional[LeaderData]:
        leaders = filter(lambda leader: leader.name, self.leaders or [])
        return next(leaders, None)

    @property
    def leader_name(self) -> Optional[str]:
        leader = self.leader
        return leader.name if leader else None

    @staticmethod
    def _is_stored_older(stored_ts: Optional[datetime], actual_date: Optional[date]) -> bool:
        return (
            stored_ts is not None
            and actual_date is not None
            and stored_ts.timetuple() < actual_date.timetuple()
        )

    @staticmethod
    def _format_phone(phone: Optional[PhoneData]) -> str:
        return f'+7 {phone.code} {phone.number}' if phone else ''

    def get_patched_organization(self,
                                 organization: Optional[OrganizationData],
                                 stored_ts: Optional[datetime]) -> Optional[OrganizationData]:
        spark_organization = self.organization_data.organization
        return (
            spark_organization
            if not organization or self._is_stored_older(stored_ts, self.organization_data.actual_date)
            else OrganizationData(
                type=organization.type or spark_organization.type,
                name=pick_opt_str(organization.name, spark_organization.name),
                english_name=pick_opt_str(organization.english_name, spark_organization.english_name),
                full_name=pick_opt_str(organization.full_name, spark_organization.full_name),
                inn=pick_opt_str(organization.inn, spark_organization.inn),
                kpp=pick_opt_str(organization.kpp, spark_organization.kpp),
                ogrn=pick_opt_str(organization.ogrn, spark_organization.ogrn),
            )
        )

    def get_patched_persons(self,
                            persons: Optional[List[PersonData]],
                            leader_name: str,
                            leader_surname: str,
                            leader_patronymic: Optional[str],
                            stored_ts: Optional[datetime]) -> List[PersonData]:
        persons = persons or []
        leader = self.leader
        if not leader:
            return persons

        ceo = next(
            filter(lambda person: person.type == PersonType.CEO, persons),
            None
        )
        if ceo:
            phone = self.phone
            is_stored_older = self._is_stored_older(stored_ts, phone.verification_date if phone else None)
            if not ceo.phone or is_stored_older:
                ceo.phone = self._format_phone(phone)

            is_stored_empty = all([not getattr(ceo, attr) for attr in ['name', 'surname', 'patronymic', 'birth_date']])
            if is_stored_empty or self._is_stored_older(stored_ts, leader.actual_date):
                ceo.name = leader_name
                ceo.surname = leader_surname
                ceo.patronymic = leader_patronymic
                ceo.birth_date = leader.birth_date
            return persons

        return persons + [
            PersonData(
                type=PersonType.CEO,
                name=leader_name,
                surname=leader_surname,
                patronymic=leader_patronymic,
                birth_date=leader.birth_date,
                email='',
                phone=self._format_phone(self.phone),
            )
        ]

    def get_patched_addresses(self,
                              addresses: Optional[List[AddressData]],
                              stored_ts: Optional[datetime]) -> List[AddressData]:
        addresses = addresses or []
        if not self.addresses:
            return addresses

        spark_address = self.addresses[0]
        legal_address = next(
            filter(lambda address: address.type == 'legal', addresses),
            None
        )
        if legal_address:
            attrs = ['city', 'country', 'home', 'street', 'zip']
            is_stored_empty = all([not getattr(legal_address, attr) for attr in attrs])
            is_stored_older = self._is_stored_older(stored_ts, spark_address.actual_date)
            if is_stored_empty or is_stored_older:
                for attr in attrs:
                    setattr(legal_address, attr, getattr(spark_address.address, attr))
        else:
            addresses.append(spark_address.address)
        return addresses


def ns_tag(ns: Union[str, XmlNs], tag: Union[str, SoapTag]) -> str:
    patched_ns = ns
    if isinstance(ns, XmlNs):
        patched_ns = ns.value

    patched_tag = tag
    if isinstance(tag, SoapTag):
        patched_tag = tag.value

    return f'{{{patched_ns}}}{patched_tag}'
