import xml.etree.ElementTree as xml
from typing import Dict, Optional

from sendr_interactions.clients.spark.entities import SoapTag, SparkMethod, XmlNs, ns_tag


class SparkRequest:
    @staticmethod
    def _request(method: SparkMethod, params: Dict[str, str]) -> str:
        for prefix, uri in XmlNs.NAMESPACES.items():  # type: ignore
            xml.register_namespace(prefix, uri)

        root = xml.Element(ns_tag(XmlNs.SOAP, SoapTag.ENVELOPE))
        body = xml.SubElement(root, ns_tag(XmlNs.SOAP, SoapTag.BODY))
        req_method = xml.SubElement(body, ns_tag(XmlNs.IFAX, method.value))

        for param_name, param_value in params.items():
            param = xml.SubElement(req_method, ns_tag(XmlNs.IFAX, param_name))
            param.text = str(param_value)
        return xml.tostring(root, encoding='utf-8', xml_declaration=True)

    @classmethod
    def auth(cls, login: str, password: str) -> str:
        return cls._request(method=SparkMethod.AUTH, params={'Login': login, 'Password': password})

    @classmethod
    def end(cls) -> str:
        return cls._request(method=SparkMethod.END, params={})

    @classmethod
    def get_entrepreneur(cls, inn: str) -> str:
        return cls._request(method=SparkMethod.GET_ENTREPRENEUR, params={'inn': inn})

    @classmethod
    def get_company(cls, inn: str, spark_id: Optional[int] = None) -> str:
        params = {'inn': inn}
        if spark_id:
            params['sparkId'] = str(spark_id)
        return cls._request(method=SparkMethod.GET_COMPANY, params=params)
