import re
from datetime import date, datetime
from typing import Any, List, Optional

import xmltodict

from sendr_interactions.clients.spark.entities import (
    AddressData, LeaderData, MerchantType, OkvedItem, OrganizationData, PhoneData, SoapTag, SparkAddressData, SparkData,
    SparkMethod, SparkOrganizationData, XmlNs, ns_tag
)
from sendr_interactions.clients.spark.exceptions import (
    BaseSparkError, SparkAuthError, SparkGetInfoError, SparkSessionEndError
)

sentinel = object()


class SparkResponse:
    @staticmethod
    def _get_tag(data: dict, tag: str, ns_alias: XmlNs, default: Optional[Any] = sentinel) -> Optional[Any]:
        ns_alias_value = ns_alias.value
        tag_alias = f'{ns_alias_value}:{tag}'
        tag_ns = ns_tag(XmlNs.NAMESPACES[ns_alias_value], tag)  # type: ignore

        if tag in data:
            return data[tag]
        elif tag_ns in data:
            return data[tag_ns]
        elif tag_alias in data:
            return data[tag_alias]
        if default is not sentinel:
            return default
        raise KeyError()

    @classmethod
    def _get_soap_tag(cls, data: dict, tag: SoapTag) -> Any:
        return cls._get_tag(data, tag.value, XmlNs.ALIAS_SOAP, None)

    @classmethod
    def _get_ifax_tag(cls, data: dict, tag: str, default: Optional[Any] = sentinel) -> Any:
        return cls._get_tag(data, tag, XmlNs.ALIAS_IFAX, default)

    @staticmethod
    def _cast_bool(data: str) -> bool:
        ldata = data.lower()
        return ldata == 'true' or ldata == '1'

    @classmethod
    def _soap_response(cls, method: SparkMethod, data: bytes) -> dict:
        parsed = xmltodict.parse(data, dict_constructor=dict)
        root = cls._get_soap_tag(parsed, SoapTag.ENVELOPE)
        body = cls._get_soap_tag(root, SoapTag.BODY)

        interaction_method = method.value
        response_root_tag = f'{interaction_method}Response'
        response_result_tag = f'{interaction_method}Result'
        response = cls._get_ifax_tag(body, response_root_tag)
        result = cls._get_ifax_tag(response, response_result_tag)
        if not cls._cast_bool(result):
            raise BaseSparkError(message=result)

        return response

    @classmethod
    def _report_data(cls, soap_data: dict) -> dict:
        xml_data = cls._get_ifax_tag(soap_data, 'xmlData')
        parsed_soap_data = xmltodict.parse(xml_data, dict_constructor=dict)

        response = cls._get_ifax_tag(parsed_soap_data, 'Response')
        data = cls._get_ifax_tag(response, 'Data')
        report = cls._get_ifax_tag(data, 'Report')

        # SPARK should return only one report, because company selected by tuple: inn + spark_id.
        # In other case, select only first report. See details in PAYBACK-821
        return report[0] if isinstance(report, list) else report

    @staticmethod
    def _parse_spark_date(data: str) -> date:
        return date.fromtimestamp(datetime.strptime(data, '%Y-%m-%d').timestamp())

    @staticmethod
    def _parse_spark_date_opt(data: Optional[str]) -> Optional[date]:
        return SparkResponse._parse_spark_date(data) if data else None

    @classmethod
    def _as_list(cls, data: dict, tag: str, default: Optional[Any]) -> Optional[List[Any]]:
        # ugly hack: xmltodict convert single child to dict and multiple children to list
        #            so always convert children to list to unify interface
        items = cls._get_ifax_tag(data, tag, default)
        if items is None:
            return None
        if isinstance(items, list):
            return items
        return [items]

    @classmethod
    def _parse_okved_list(cls, data: Optional[dict]) -> Optional[List[OkvedItem]]:
        if data is None:
            return None
        okved_list = cls._as_list(data, 'OKVED', None)
        if okved_list is None:
            return None
        result: List[OkvedItem] = []
        for item in okved_list:
            result.append(OkvedItem(
                main=bool(item.get('@IsMain', False)),
                code=item.get('@Code', None),
                name=item.get('@Name', None)
            ))
        return result

    @classmethod
    def _parse_leaders(cls, data: Optional[dict]) -> Optional[List[LeaderData]]:
        if data is None:
            return None
        leader_list = cls._as_list(data, 'Leader', None)
        if leader_list is None:
            return None
        result: List[LeaderData] = []
        for leader in leader_list:
            result.append(LeaderData(
                name=leader.get('@FIO', None),
                position=leader.get('@Position', None),
                inn=leader.get('@INN', None),
                actual_date=cls._parse_spark_date_opt(leader.get('@ActualDate', None)),
            ))
        return result

    @staticmethod
    def _format_home_for_address(data: dict) -> Optional[str]:
        building, housing, block = [
            data.get(key, '')
            for key in ['@BuildingNumber', '@Housing', '@Block']
        ]
        housing_sep = '/' if building and (housing or block) else ''
        block_sep = 'с' if housing and block else ''
        home = ''.join([building, housing_sep, housing, block_sep, block])
        return home if home else None

    @staticmethod
    def _cut_address_prefix(address: str) -> str:
        return re.sub(r'^[a-zA-Zа-яА-ЯёЁ-]+\.\s+', '', address)

    @staticmethod
    def _clean_latin_string(string: Optional[str]) -> Optional[str]:
        return re.sub(r'[^a-zA-Z0-9_\.\s-]+', '', string) if string else string

    @classmethod
    def _parse_addresses(cls, data: Optional[dict]) -> Optional[List[SparkAddressData]]:
        if data is None:
            return None
        address_list = cls._as_list(data, 'Address', None)
        if address_list is None:
            return None
        result: List[SparkAddressData] = []
        for address in address_list:
            city, street, post_code = [
                address.get(key, '')
                for key in ['@City', '@StreetName', '@PostCode']
            ]
            home = cls._format_home_for_address(address)
            if not street and home:
                street, home = home, street
            if any([city, home, street, post_code]):
                result.append(
                    SparkAddressData(
                        address=AddressData(
                            type='legal',
                            city=cls._cut_address_prefix(city),
                            country='RUS',
                            street=cls._cut_address_prefix(street),
                            home=home,
                            zip=post_code
                        ),
                        actual_date=cls._parse_spark_date_opt(address.get('@ActualDate', None)),
                    )
                )
        return result if result else None

    @classmethod
    def _parse_phones(cls, data: Optional[dict]) -> Optional[List[PhoneData]]:
        if data is None:
            return None
        phone_list = cls._as_list(data, 'Phone', None)
        if phone_list is None:
            return None
        result: List[PhoneData] = []
        for phone in phone_list:
            result.append(PhoneData(
                code=phone.get('@Code', None),
                number=phone.get('@Number', None),
                verification_date=cls._parse_spark_date_opt(phone.get('@VerificationDate', None))
            ))
        return result

    @classmethod
    def _parse_is_active(cls, data: Optional[dict]) -> Optional[bool]:
        if data is None:
            return None
        is_active_str: str = data.get('@IsActing', '')
        return cls._cast_bool(is_active_str)

    @classmethod
    def check_auth_response(cls, data: bytes) -> None:
        try:
            cls._soap_response(SparkMethod.AUTH, data)
        except BaseSparkError as e:
            raise SparkAuthError(message=e.message)

    @classmethod
    def check_end_response(cls, data: bytes) -> None:
        try:
            cls._soap_response(SparkMethod.END, data)
        except BaseSparkError as e:
            raise SparkSessionEndError(message=e.message)

    @classmethod
    def parse_entrepreneur_response(cls, data: bytes) -> SparkData:
        try:
            soap_data = cls._soap_response(SparkMethod.GET_ENTREPRENEUR, data)
        except BaseSparkError as e:
            raise SparkGetInfoError(message=e.message)

        report = cls._report_data(soap_data)

        reg_date = cls._get_ifax_tag(report, 'DateFirstReg', None) or \
            cls._get_ifax_tag(report, 'DateReg', None)
        full_name_value = cls._get_ifax_tag(report, 'FullNameRus', None)

        return SparkData(
            spark_id=cls._get_ifax_tag(report, 'SparkID'),
            registration_date=cls._parse_spark_date_opt(reg_date),
            organization_data=SparkOrganizationData(
                organization=OrganizationData(
                    type=MerchantType.IP,
                    full_name=full_name_value,
                    inn=cls._get_ifax_tag(report, 'INN', None),
                    ogrn=cls._get_ifax_tag(report, 'OGRNIP', None)
                ),
                actual_date=cls._parse_spark_date_opt(report.get('@ActualDate', None)),
            ),
            okved_list=cls._parse_okved_list(cls._get_ifax_tag(report, 'OKVED2List', None)),
            leaders=None if full_name_value is None else [
                LeaderData(
                    name=full_name_value,
                    birth_date=cls._parse_spark_date_opt(cls._get_ifax_tag(report, 'BirthDate', None)),
                )
            ],
            phones=cls._parse_phones(cls._get_ifax_tag(report, 'PhoneList', None)),
            active=cls._parse_is_active(cls._get_ifax_tag(report, 'Status', None)),
        )

    @classmethod
    def parse_company_response(cls, data: bytes) -> SparkData:
        try:
            soap_data = cls._soap_response(SparkMethod.GET_COMPANY, data)
        except BaseSparkError as e:
            raise SparkGetInfoError(message=e.message)

        report = cls._report_data(soap_data)

        return SparkData(
            spark_id=cls._get_ifax_tag(report, 'SparkID'),
            registration_date=cls._parse_spark_date_opt(cls._get_ifax_tag(report, 'DateFirstReg', None)),
            organization_data=SparkOrganizationData(
                organization=OrganizationData(
                    type=MerchantType.OOO,
                    name=cls._get_ifax_tag(report, 'ShortNameRus', None),
                    english_name=cls._clean_latin_string(cls._get_ifax_tag(report, 'ShortNameEn', None)),
                    full_name=cls._get_ifax_tag(report, 'FullNameRus', None),
                    inn=cls._get_ifax_tag(report, 'INN', None),
                    kpp=cls._get_ifax_tag(report, 'KPP', None),
                    ogrn=cls._get_ifax_tag(report, 'OGRN', None)
                ),
                actual_date=cls._parse_spark_date_opt(report.get('@ActualDate', None)),
            ),
            okved_list=cls._parse_okved_list(cls._get_ifax_tag(report, 'OKVED2List', None)),
            leaders=cls._parse_leaders(cls._get_ifax_tag(report, 'LeaderList', None)),
            addresses=cls._parse_addresses(cls._get_ifax_tag(report, 'LegalAddresses', None)),
            phones=cls._parse_phones(cls._get_ifax_tag(report, 'PhoneList', None)),
            active=cls._parse_is_active(cls._get_ifax_tag(report, 'Status', None)),
        )
