import json
from typing import Any, ClassVar, Collection, Dict, Optional

from aiohttp import ClientResponse, ContentTypeError

from sendr_interactions.base import AbstractInteractionClient
from sendr_interactions.clients.startrek.entities import Issue, Link
from sendr_interactions.clients.startrek.exceptions import BaseStartrekError
from sendr_interactions.clients.startrek.schemas import CreateIssueRequestSchema, IssueSchema
from sendr_utils import without_none


class AbstractStartrekClient(AbstractInteractionClient[Dict[str, Any]]):
    SERVICE: ClassVar[str] = 'startrek'

    BASE_URL: ClassVar[str]
    OAUTH_TOKEN: ClassVar[str]

    def _get_session_kwargs(self) -> dict:
        kwargs = super()._get_session_kwargs()
        headers = kwargs.setdefault('headers', {})
        headers['Authorization'] = f'OAuth {self.OAUTH_TOKEN}'
        return kwargs

    async def _handle_response_error(self, response: ClientResponse) -> None:
        error_code = response.status
        exc_cls = BaseStartrekError.get_exception_by_code(error_code)
        try:
            data = await response.json()
            message = json.dumps(data.get('errorMessages', 'No errorMessage present'))
        except (ContentTypeError, json.JSONDecodeError):
            message = 'Unable to decode error json'
        await self._try_log_error_response_body(response)
        raise exc_cls(method=response.method, message=message, service=self.SERVICE)

    async def create_issue(
        self,
        queue: str,
        summary: str,
        description: Optional[str] = None,
        parent: Optional[str] = None,
        links: Collection[Link] = frozenset(),
        attachment_ids: Collection[str] = frozenset(),
        tags: Collection[str] = frozenset(),
        unique: Optional[str] = None,
    ) -> Issue:
        """Creates a new issue.

        Docs: https://wiki.yandex-team.ru/tracker/api/issues/create/
        queue -- queue key (e.g. TEST, PAYBACK, etc).
        summary -- issue header.
        unique -- idempotency key. Note: next_task_key sequential number is still incremented on each conflict.
        """
        url = self.endpoint_url('/v2/issues')
        json_data = without_none({
            'queue': queue,
            'summary': summary,
            'description': description,
            'parent': parent,
            'unique': unique,
            'links': links,
            'attachmentIds': attachment_ids,
            'tags': tags,
        })
        response = await self.post(
            'create_issue',
            url,
            json=CreateIssueRequestSchema().dump(json_data).data,
        )
        return IssueSchema().load(response).data

    async def update_issue(
        self,
        issue_id: str,
        summary: Optional[str] = None,
        description: Optional[str] = None,
    ) -> Issue:
        """Updates an issue. Semantic is PATCH.

        Docs: https://st-api.yandex-team.ru/docs/#operation/updateIssueApiV2
        issue_id -- issue.id or issue.key (e.g. YANDEXPAYTEST-17 OR 60d501d28c3625165e149687)
        """
        self.assert_string_urlsafe_for_path(issue_id)

        url = self.endpoint_url(f'/v2/issues/{issue_id}')
        json_data = without_none({
            'description': description,
            'summary': summary,
        })
        response = await self.patch(
            'update_issue',
            url,
            json=CreateIssueRequestSchema().dump(json_data).data,
        )
        return IssueSchema().load(response).data

    async def execute_issue_transition(
        self,
        issue_id: str,
        transition_id: str,
        resolution_key: Optional[str] = None,
    ) -> None:
        """Executes status transition.

        Docs: https://wiki.yandex-team.ru/tracker/api/issues/transitions-execute/
        Related docs:
            Transitions list: https://wiki.yandex-team.ru/tracker/api/issues/transitions-list/
        issue_id -- issue.id or issue.key (e.g. YANDEXPAYTEST-17 OR 60d501d28c3625165e149687)
        transition_id -- id of the transition, e.g. close OR need_info
        """
        self.assert_string_urlsafe_for_path(issue_id)
        self.assert_string_urlsafe_for_path(transition_id)

        url = self.endpoint_url(f'/v2/issues/{issue_id}/transitions/{transition_id}/_execute')
        json_data = without_none({
            'resolution': resolution_key,
        })

        await self.post(
            'execute_transition',
            url,
            json=json_data,
        )

    async def get_issue_info(self, issue_id: str) -> Issue:
        """Get the issue info.

        Docs: https://wiki.yandex-team.ru/tracker/api/issues/get/
        issue_id -- issue.id or issue.key (e.g. YANDEXPAYTEST-17 OR 60d501d28c3625165e149687)
        """

        self.assert_string_urlsafe_for_path(issue_id)
        url = self.endpoint_url(f'/v2/issues/{issue_id}')
        response = await self.get('get_issue_status', url)
        return IssueSchema().load(response).data
