from __future__ import annotations

from typing import Any, ClassVar, Dict, Optional, Type

from sendr_interactions import exceptions as interaction_errors


class BaseStartrekError(interaction_errors.BaseInteractionError):
    """
    https://wiki.yandex-team.ru/tracker/api/#oshibki
    """
    _exceptions_by_code: ClassVar[Dict[int, Type[BaseStartrekError]]] = {}

    def __init__(
        self, *,
        method: str,
        service: str,
        message: Optional[str] = None,
        params: Optional[Dict[str, Any]] = None,
    ) -> None:
        if params is not None:
            self.params = params
        super().__init__(service=service, method=method, message=message)

    def __init_subclass__(cls) -> None:
        code: int = getattr(cls, 'HTTP_STATUS_CODE', None)
        if code is not None:
            assert code not in BaseStartrekError._exceptions_by_code
            BaseStartrekError._exceptions_by_code[code] = cls

    @classmethod
    def get_exception_by_code(cls, http_status_code: int) -> Type[BaseStartrekError]:
        return cls._exceptions_by_code.get(http_status_code, UnknownStartrekError)


class UnknownStartrekError(BaseStartrekError):
    pass


class ForbiddenStartrekError(BaseStartrekError):
    HTTP_STATUS_CODE = 403


class NotFoundStartrekError(BaseStartrekError):
    HTTP_STATUS_CODE = 404


class AlreadyExistsStartrekError(BaseStartrekError):
    HTTP_STATUS_CODE = 409


class BadFormatStartrekError(BaseStartrekError):
    HTTP_STATUS_CODE = 400
