from typing import Any, Dict

from aiohttp import ClientResponse

from sendr_interactions.base import AbstractInteractionClient
from sendr_interactions.clients.sup.entities import PushRequest, PushResponse, RegistrationsResponse
from sendr_interactions.clients.sup.schemas import PushRequestSchema, PushResponseSchema, RegistrationsResponseSchema
from sendr_interactions.exceptions import InteractionResponseError


class AbstractSUPClient(AbstractInteractionClient[Dict[str, Any]]):
    """Клиент для сервиса SUP – рассылка пуш уведомлений

    `Документация к API <https://doc.yandex-team.ru/sup-api/concepts/index.html>`_
    """

    SERVICE = 'sup'

    INTERACTION_METHOD_REGISTER_INSTALLATION = 'register_installation'

    def _get_session_kwargs(self) -> Dict[str, Any]:
        kwargs = super()._get_session_kwargs()
        kwargs.setdefault('headers', {})['Authorization'] = f'OAuth {self.OAUTH_TOKEN}'
        return kwargs

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> Dict[str, Any]:
        if response.status >= 400:
            await self._handle_response_error(response)
        if interaction_method == self.INTERACTION_METHOD_REGISTER_INSTALLATION:
            return {}
        return await self._format_response(response)

    async def _handle_response_error(self, response: ClientResponse) -> None:
        await self._try_log_error_response_body(response)
        raise InteractionResponseError(
            status_code=response.status,
            response_status=response.reason,
            method=response.method,
            service=self.SERVICE,
            params=await response.json(),
        )

    async def send_push(self, push: PushRequest) -> PushResponse:
        """Отправка пуша

        `Описание <https://doc.yandex-team.ru/sup-api/concepts/push-create/index.html>`_
        """
        endpoint_url = self.endpoint_url('pushes')
        data, error = PushRequestSchema().dump(push)
        if error:
            raise InteractionResponseError(
                status_code=500,
                response_status='failed to dump request payload',
                method='POST',
                service=self.SERVICE,
                params=error,
            )
        response = await self.post(
            interaction_method='pushes',
            url=endpoint_url,
            params={'dry_run': 0},
            json=data,
        )
        return PushResponseSchema().load_one(response)

    async def get_installation(self, install_id: str) -> RegistrationsResponse:
        """Получение сведений об установке

        `Описание <https://doc.yandex-team.ru/sup-api/concepts/user-registration-check.html>`_
        """
        self.assert_string_urlsafe_for_path(install_id)
        endpoint_url = self.endpoint_url(f'registrations/{install_id}/')
        response = await self.get(
            interaction_method='registrations',
            url=endpoint_url,
        )
        return RegistrationsResponseSchema().load_one(response)

    async def register_installation(
        self,
        app_id: str,
        app_version: str,
        hardware_id: str,
        push_token: str,
        platform: str,
        device_name: str,
        zone_id: str,
        notify_disabled: bool,
        active: bool,
        install_id: str,
        device_id: str,
        vendor_device_id: str,
        is_huawei: bool,
    ) -> None:
        endpoint_url = self.endpoint_url('v2/registrations')
        body = {
            'appId': app_id,
            'appVersion': app_version,
            'hardwareId': hardware_id,
            'platform': platform,
            'deviceName': device_name,
            'zoneId': zone_id,
            'notifyDisabled': notify_disabled,
            'active': active,
            'installId': install_id,
            'deviceId': device_id,
            'vendorDeviceId': vendor_device_id,
        }
        if not is_huawei:
            body['pushToken'] = push_token
        else:
            body['pushTokens'] = {
                'hms': push_token,
            }
        await self.post(
            interaction_method=self.INTERACTION_METHOD_REGISTER_INSTALLATION,
            url=endpoint_url,
            json=body,
        )
