from inflection import camelize, underscore
from marshmallow import fields, post_dump, pre_load

from sendr_interactions.clients.sup.entities import (
    Data, Hypothesis, Notification, Push, PushRequest, PushResponse, RegistrationsResponse, ThrottlePolicies
)
from sendr_utils.schemas.dataclass import BaseDataClassSchema


class DataSchema(BaseDataClassSchema[Data]):
    SKIP_NONE = True

    push_id = fields.String()
    content_id = fields.String()
    stat_id = fields.String()
    transit_id = fields.String()
    push_uri = fields.String()
    push_action = fields.String()
    topic_push = fields.String()


class NotificationSchema(BaseDataClassSchema[Notification]):
    SKIP_NONE = True

    title = fields.String()
    body = fields.String()
    icon = fields.String()
    iconId = fields.String()
    link = fields.String()


class PushSchema(BaseDataClassSchema[Push]):
    SKIP_NONE = True

    notification = fields.Nested(NotificationSchema)


class HypothesisSchema(BaseDataClassSchema[Hypothesis]):
    SKIP_NONE = True

    id = fields.String()
    push = fields.Nested(PushSchema)


class ThrottlePoliciesSchema(BaseDataClassSchema[ThrottlePolicies]):
    SKIP_NONE = True

    install_id = fields.String()
    device_id = fields.String()
    content_id = fields.String()


class PushRequestSchema(BaseDataClassSchema[PushRequest]):
    SKIP_NONE = True

    receiver = fields.List(fields.String(), required=True)
    ttl = fields.String()
    schedule = fields.String()
    adjust_time_zone = fields.Bool()
    send_time_frame = fields.String()
    data = fields.Nested(DataSchema)
    notification = fields.Nested(NotificationSchema)
    hypotheses = fields.Nested(HypothesisSchema, many=True)
    project = fields.String()
    max_expected_receivers = fields.String()
    android_features = fields.Dict(keys=fields.Str(), values=fields.Str())
    ios_features = fields.Dict(keys=fields.Str(), values=fields.Str())
    browser_features = fields.Dict(keys=fields.Str(), values=fields.Str())
    throttle_policies = fields.Nested(ThrottlePoliciesSchema)
    actions = fields.Dict(keys=fields.Str(), values=fields.Str())
    is_data_only = fields.Bool()
    spread_interval = fields.String()


class PushResponseSchema(BaseDataClassSchema[PushResponse]):
    SKIP_NONE = True

    id = fields.String()
    receiver = fields.List(fields.String())
    data = fields.Dict(keys=fields.Str(), values=fields.Str())
    request_time = fields.Integer()

    @pre_load
    def to_snakecase(self, data, **kwargs):
        return {underscore(key): value for key, value in data.items()}

    @post_dump
    def to_camelcase(self, data, **kwargs):
        return {camelize(key, False): value for key, value in data.items()}


class RegistrationsResponseSchema(BaseDataClassSchema[RegistrationsResponse]):
    SKIP_NONE = True

    app_id = fields.String()
    app_version = fields.String()
    hardware_id = fields.String()
    push_token = fields.String()
    push_tokens = fields.Dict(keys=fields.Str(), values=fields.Str())
    platform = fields.String()
    device_name = fields.String()
    zone_id = fields.String()
    notify_disabled = fields.Bool()
    active = fields.Bool()
    updated_at = fields.Int()
    install_id = fields.String()
    device_id = fields.String()

    @pre_load
    def to_snakecase(self, data, **kwargs):
        return {underscore(key): value for key, value in data.items()}

    @post_dump
    def to_camelcase(self, data, **kwargs):
        return {camelize(key, False): value for key, value in data.items()}
