from typing import Any, ClassVar, Dict, Optional, Type

from aiohttp import ClientResponse

from sendr_interactions.base import AbstractInteractionClient
from sendr_interactions.clients.zora.exceptions import get_zora_exception_by_code
from sendr_tvm.client.aiohttp import TvmHeaders, TvmTicketGetter


class AbstractZoraClient(AbstractInteractionClient[Dict[str, Any]]):
    """
    Sets zora as proxy for all requests.
    """
    SERVICE = 'zora'

    ZORA_CLIENT_NAME: ClassVar[str]
    ZORA_TVM_ID: ClassVar[int]
    ZORA_URL: ClassVar[str]
    ZORA_VERIFY_CERT: ClassVar[bool] = True
    ZORA_FOLLOW_REDIRECTS: ClassVar[bool] = False
    ZORA_ERROR_CODE_HEADER: ClassVar[str] = 'X-Yandex-Gozora-Error-Code'
    ZORA_ERROR_DESC_HEADER: ClassVar[str] = 'X-Yandex-Gozora-Error-Description'

    tvm_ticket_getter: ClassVar[Type[TvmTicketGetter]]

    async def _get_tvm_headers(self) -> TvmHeaders:
        return await self.tvm_ticket_getter.get_service_ticket_headers(self.ZORA_TVM_ID)

    async def _handle_response_error(self, response: ClientResponse) -> None:
        zora_error_code = response.headers.get(self.ZORA_ERROR_CODE_HEADER)
        if zora_error_code is not None:
            exc_type = get_zora_exception_by_code(int(zora_error_code))
            if exc_type is not None:
                await self._try_log_error_response_body(response)
                raise exc_type(
                    status_code=response.status,
                    method=response.method,
                    service=self.SERVICE,
                    message=response.headers.get(self.ZORA_ERROR_DESC_HEADER),
                )

        await super()._handle_response_error(response)

    async def _should_retry_failed_request(
        self,
        interaction_method: str,
        exc: Optional[ClientResponse] = None,
        response: Optional[ClientResponse] = None,
    ) -> bool:
        if (
            response is not None
            and response.status == 599  # used by Zora for non-standard errors
            and self.ZORA_ERROR_CODE_HEADER in response.headers
        ):
            return False
        return await super()._should_retry_failed_request(
            interaction_method=interaction_method,
            exc=exc,
            response=response,
        )

    async def _make_request(
        self, interaction_method: str, method: str, url: str, **kwargs: Any
    ) -> ClientResponse:
        headers = kwargs.get('headers', {})
        headers.update(
            {
                'X-Ya-Ignore-Certs': 'false' if self.ZORA_VERIFY_CERT else 'true',
                'X-Ya-Follow-Redirects': str(self.ZORA_FOLLOW_REDIRECTS).lower(),
                'X-Ya-Client-Id': self.ZORA_CLIENT_NAME,
                'X-Ya-Req-Id': self.request_id,
                'X-Ya-Dest-Url': url,
                **(await self._get_tvm_headers()),
            }
        )
        kwargs['headers'] = headers
        kwargs['verify_ssl'] = False

        return await super()._make_request(interaction_method, method, self.ZORA_URL, **kwargs)
