from typing import TYPE_CHECKING, Any, Callable, ClassVar, Dict, List, Optional, Type, TypedDict

from aiohttp import TCPConnector

if TYPE_CHECKING:
    from aiohttp.tracing import Trace

import yenv

from sendr_interactions.clients.sd import AbstractSDClient
from sendr_interactions.clients.sd.entities import EndpointResolveStatus
from sendr_qlog import LoggerContext
from sendr_qlog.http.aiohttp import handler_logger


class SDConfigEntry(TypedDict):
    hostname: str
    host: str
    port: str
    endpoint_set_id: Optional[str]


SDConfig = Dict[str, SDConfigEntry]


def sd(sd_config: SDConfig) -> Callable[[Type[TCPConnector]], Type[TCPConnector]]:
    """
    Декорирует коннектор для того, чтобы можно было ходить через SD.
    В interaction client'е ты указываешь dummy доменное имя, которое на самом деле является
    ключом в SD_CONFIG. Т.е. конфиграция конкретного клиента принципиально не меняется!
    Значит, если нужно переехать на балансер или с балансера, то код клиента не затрагивается.

    При разработке ходим в sd api и там резолвим endpoint_set'ы.
    В облаке ходим через goxcart.

    Пример конфига (предлагается размещать в настройках)

    YANDEX_PAY_BACKEND_URL = 'https://yandex-pay-production.ya-sd.pay.yandex.net/'
    YANDEX_PAY_PLUS_BACKEND_URL = 'https://yandex-pay-plus.ya-sd.pay.yandex.net/'
    SD_CONFIG = {
      'yandex-pay-production.ya-sd.pay.yandex.net': {
        'host': 'fdca:dfde:0001::1',
        'port': 20001,
        'hostname': 'testing.api-internal.pay.yandex.net',
        'endpoint_set_id': 'yandexpay-testing.api-internal',
      },
      'yandex-pay-plus.ya-sd.pay.yandex.net': {
        'host': 'fdca:51a4:0002::1',
        'port': 20002,
        'hostname': 'testing.api-internal.pay.yandex.net',
        'endpoint_set_id': 'yandexpay-plus-testing.api',
      },
    }

    Правила для конфига:
        * key
            Ключ должен заканчиваться на sd.[бескуковый яндексовый домен].
            Ключ содержит в себе dummy доменное имя, которое используется в interaction client'ах
            при выполнении запроса.
            Рекомендуется использовать в качестве суффикса ya-sd.[твой корневой бескуковый домен]
            Например: something.ya-sd.pay.yandex.net
            Этот домен не должен быть делегирован.
        * host
            fc00::/7 - приватные адреса, используй как хочешь.
            Предлагаю первую группу фиксировать fcda.
            Вторую группу вычислять как sha256(endpoint_set_id)[:4]
            Третью группу увеличивать инкрементально в рамках одного SD_CONFIG
        * port
            Предлагается начинать с 20001 и увеличивать инкрементально в рамках одного SD_CONFIG
        * hostname
            Если у бэкенда назначения уже есть сертификат (например в этот бэкенд ходит L7 балансер),
            то hostname следует указывать как в сертификате.
            Если же это backend-backend и доменное имя нужно выбрать, то предлагается выбирать
            бескуковый домен, затем <endpoint_set_id> перевернутый.
            Например:
                Бескуковый базовый домен: pay.yandex.net
                endpoint_set_id = yandexpay-testing.api
                Итоговый домен: api.yandexpay-testing.pay.yandex.net
    """
    def _add_sd_awareness(connector: Type[TCPConnector]) -> Type[TCPConnector]:

        if yenv.type == 'development':
            return _make_sd_aware_dev_tcp_connector(connector, sd_config)

        return _make_sd_aware_tcp_connector(connector, sd_config)

    return _add_sd_awareness


if TYPE_CHECKING:
    SDAwareTCPConnectorMixinBase = TCPConnector
else:
    SDAwareTCPConnectorMixinBase = object


class SDAwareTCPConnectorMixin(SDAwareTCPConnectorMixinBase):
    sd_config: ClassVar[SDConfig]

    async def _resolve_host(
        self, host: str, port: int, traces: Optional[List["Trace"]] = None
    ) -> List[Dict[str, Any]]:
        if host in self.sd_config:
            return await self._resolve_balancer_host(host)
        return await super()._resolve_host(host, port, traces)

    async def _resolve_balancer_host(self, host: str) -> List[Dict[str, Any]]:
        config = self.sd_config[host]
        return [
            {
                "hostname": config['hostname'],
                "host": config['host'],
                "port": config['port'],
                "family": self._family,
                "proto": 0,
                "flags": 0,
            }
        ]


class SDAwareDevTCPConnectorMixin(SDAwareTCPConnectorMixin):
    sd_client: ClassVar[AbstractSDClient]

    async def _resolve_balancer_host(self, host: str) -> List[Dict[str, Any]]:
        config = self.sd_config[host]
        if config.get('endpoint_set_id'):
            for cluster_name in ('man', 'sas', 'vla', 'iva', 'myt'):
                response = await self.sd_client.resolve_endpoints(
                    cluster_name=cluster_name,
                    endpoint_set_id=config['endpoint_set_id'],
                    client_name='dev.sendr_interactions.python.mail',
                )
                if response.resolve_status == EndpointResolveStatus.OK:
                    for item in response.endpoint_set.endpoints:
                        if item.ready:
                            return [
                                {
                                    "hostname": config['hostname'],
                                    "host": item.ip6_address,
                                    "port": item.port,
                                    "family": self._family,
                                    "proto": 0,
                                    "flags": 0,
                                }
                            ]

        raise OSError('Host is not resolved')


def _make_sd_aware_tcp_connector(
    base_connector_cls: Type[TCPConnector],
    sd_config: Dict[str, SDConfigEntry]
) -> Type[TCPConnector]:
    orig_sd_config = sd_config

    # ignore reason: https://github.com/python/mypy/issues/2477#issuecomment-551193110
    class SDAwareTCPConnector(SDAwareTCPConnectorMixin, base_connector_cls):  # type: ignore
        sd_config = orig_sd_config

    return SDAwareTCPConnector


def _make_sd_aware_dev_tcp_connector(
    base_connector_cls: Type[TCPConnector],
    sd_config: Dict[str, SDConfigEntry]
) -> Type[TCPConnector]:
    class SDClient(AbstractSDClient):
        CONNECTOR = base_connector_cls()
        DEBUG = True
        REQUEST_RETRY_TIMEOUTS = ()

    orig_sd_config = sd_config

    # ignore reason: https://github.com/python/mypy/issues/2477#issuecomment-551193110
    class SDAwareDevTCPConnector(SDAwareDevTCPConnectorMixin, base_connector_cls):  # type: ignore
        sd_config = orig_sd_config
        sd_client = SDClient(
            logger=LoggerContext(handler_logger),
            request_id='dev.sendr_interactions.python.mail'
        )

    return SDAwareDevTCPConnector
