from collections import defaultdict
from typing import Dict, Protocol


class RetryBudgetProtocol(Protocol):
    """Бюджет ретраев

    Необходим, чтобы снизить нагрузку на сервер в случае его отказа

    https://a.yandex-team.ru/arc/trunk/arcadia/mail/ymod_httpclient/doc/cluster_client/retries.md
    """
    def success(self, key: str) -> None:
        """Вызывается при успешном выполнении запроса"""
        raise NotImplementedError

    def fail(self, key: str) -> None:
        """Вызывается при неуспешном выполнении запроса"""
        raise NotImplementedError

    def can_retry(self, key: str) -> bool:
        """Проверка возможности повторной попытки запроса"""
        raise NotImplementedError


class UnlimitedRetryBudget:
    """Бюджет, который не ограничивает перезапросы

    Нужен как класс по умолчанию, чтобы не изменять поведение клиента
    """
    def success(self, key: str) -> None:
        pass

    def fail(self, key: str) -> None:
        pass

    def can_retry(self, key: str) -> bool:
        return True


class RetryBudget:
    """Бюджет ретраев, основан на спецификации GRPC_

    Бюджетируется hostname

    .. _GRPC: https://github.com/grpc/proposal/blob/master/A6-client-retries.md#detailed-design
    """

    _items: Dict[str, float]

    def __init__(self, *, max_tokens: float = 10, token_ratio: float = 0.1):
        self._max_tokens = max_tokens
        self._token_ratio = token_ratio
        self._threshold = max_tokens / 2
        self._items = defaultdict(lambda: max_tokens)

    def success(self, key: str) -> None:
        token_count = self._items[key]
        self._items[key] = min(self._max_tokens, token_count + self._token_ratio)

    def fail(self, key: str) -> None:
        token_count = self._items[key]
        self._items[key] = max(.0, token_count - 1)

    def can_retry(self, key: str) -> bool:
        return self._items[key] >= self._threshold
