import random
import string
from collections import defaultdict
from contextlib import contextmanager
from datetime import datetime
from decimal import Decimal
from enum import EnumMeta
from typing import Any, List, Optional

import pytest

from sendr_utils import get_subclasses as base_get_subclasses
from sendr_utils import utcnow

__all__ = (
    'coromock',
    'issued_rand',
    'mock_action',
    'noop',
    'noop_manager',
    'get_subclasses',
    'randbool',
    'randdate',
    'randdecimal',
    'randitem',
    'randmail',
    'randn',
    'rands',
    'randslice',
    'randurl',
    'returned',
    'request_id',
    'unique_rand',
    'unixtime',
)


@pytest.fixture
def mock_action(mocker):
    """Mocks sendr_core + sendr_taskqueue action.
    Replaces constructor with a mock and .run() with a dummy function returning `action_result`.
    """

    def _inner(action_cls, action_result=None, action_func=None):
        async def run(self):
            if (
                isinstance(action_result, Exception)
                or isinstance(action_result, type) and issubclass(action_result, Exception)
            ):
                raise action_result
            return action_result

        async def run_async(self):
            return None

        mocker.patch.object(action_cls, 'run', run if action_func is None else action_func)
        if hasattr(action_cls, 'run_async'):
            mocker.patch.object(action_cls, 'run_async', run_async)
        return mocker.patch.object(action_cls, '__init__', mocker.Mock(return_value=None))

    return _inner


@pytest.fixture
def unixtime():
    return lambda: int(utcnow().timestamp())


@pytest.fixture(scope='session')
def issued_rand():
    return defaultdict(set)


@pytest.fixture()
def unique_rand(issued_rand):
    def _inner(func, *args, **kwargs):
        assert func.__name__ != '<lambda>'
        key = f'{kwargs.pop("basket", "default")}-{func.__module__}.{func.__name__}'

        while True:
            result = func(*args, **kwargs)
            if result not in issued_rand[key]:
                issued_rand[key].add(result)
                return result

    return _inner


@pytest.fixture
def randn():
    def _randn_inner(**kwargs):
        from_val, to_val = kwargs.get('min', 1), kwargs.get('max', 10 ** 9)
        assert from_val <= to_val
        return random.randint(from_val, to_val) if from_val != to_val else from_val

    return _randn_inner


@pytest.fixture
def rands():
    def _rands_inner(**kwargs):
        return ''.join(random.choices(string.ascii_letters, k=kwargs.get('k', 12)))

    return _rands_inner


@pytest.fixture
def randbool():
    def _randbool_inner(**kwargs):
        return random.choice((True, False))

    return _randbool_inner


@pytest.fixture
def randitem():
    def _randitem_inner(items, **kwargs):
        if isinstance(items, (EnumMeta, frozenset)):
            return random.choice(list(items))
        return random.choice(items)

    return _randitem_inner


@pytest.fixture
def randmail(rands):
    return lambda: f'{rands()}@{rands()}.ru'


@pytest.fixture
def randurl(rands):
    return lambda: f'http{random.choices(("", "s"))}://{rands()}.{random.choices(("com", "ru"))}/{rands()}'


@pytest.fixture
def randdecimal(randn):
    def _randdecimal(**kwargs):
        return Decimal(f'{randn(**kwargs)}.{str(randn())[:2]}')

    return _randdecimal


@pytest.fixture
def randslice(randn):
    def _randslice(array: List, min_length: Optional[int] = None) -> List:
        n = len(array)
        if n < 2:
            return array

        if min_length:
            min_length = min(min_length, n)

        i = randn(min=0, max=n - 1) if min_length is None else randn(min=0, max=n - min_length)
        j = randn(min=i + (1 if min_length is None else min_length), max=n)

        return array[i:j]

    return _randslice


@pytest.fixture
def randdate(randn, unixtime):
    def _randdate() -> datetime:
        now_ts = unixtime()
        diff = 86400

        return datetime.fromtimestamp(randn(min=now_ts - diff, max=now_ts + diff))

    return _randdate


@pytest.fixture
def coromock(mocker):
    def _inner(result: Optional[Any] = None, exc: Optional[Any] = None) -> Any:
        async def coro(*args: Any, **kwargs: Any) -> Optional[Any]:
            if exc:
                raise exc
            return result

        return mocker.Mock(side_effect=coro)

    return _inner


@pytest.fixture
def noop():
    def inner(*args, **kwargs):
        pass

    return inner


@pytest.fixture
def noop_manager():
    @contextmanager
    def inner(*args, **kwargs):
        yield

    return inner


@pytest.fixture
async def returned(returned_func):
    return await returned_func()


@pytest.fixture
def request_id():
    return 'unittest-requestid'


@pytest.fixture
def get_subclasses():
    def _get_subclasses(base_cls):
        return base_get_subclasses(base_cls, '.test_')
    return _get_subclasses
