"""
Служебный модуль-плагин с определениями pytest-хуков для выявления коллизий тестов по nodeid.
Чтобы задействовать плагин, нужно либо импортировать pytest-хуки в базовый conftest.py как есть,
либо добавить к существующим хукам с соответсвтующим названием.

Плагин добавляет опцию --detect-conflicts=(yes|no). При обнаружении конфликта будет выведена информация
о конфликтующих тестах и выброшен RuntimeError, чтобы гарантированно уронить тестовую сессию.
Проверка на конфликт происходит после сбора всей коллекции тестов, но перед запуском тестов.

Сочетание с опцией --collect-only позволяет проверить тесты на конфликты, гарантированно не запуская их.
В случае конфликта до запуска тестов дело не дойдет, но если конфликтов нет, то опция позволит
предотвратить запуск.

Опция --print-metafunc-ids покажет все nodeid тестовых функций без учета параметризации, например:
    package/tests/unit/test_something.py::TestAnything::test_hello_world

Опция --no-run - замена опции --collect-only без вывода какой-то доп. информации.

Зачем все это нужно:
После запуска `ya make -tt` во время генерации отчета некорректно обрабатываются nodeid тестов,
которые расположены внутри вложенных друг в друга классов. Утилиты составления отчета о тестировании
не учитывают промежуточные классы для построения идентификатора теста и корректный nodeid
    test.py::TestA::TestB::test
превращается в некорректный
    test.py::TestA::test

Из-за данной особенности в составлении отчета упавший тест может быть заслонен успешным тестом,
если у обоих тестов совпали аркадийные идентификаторы. Никаких предупреждений или сообщений о коллизиях
при этом не будет.

Пожалуй, наличие подобных конфликтов является признаком плохо организованных тестов,
поэтому за ними следует следить явно и стремиться не писать тесты с глубокой вложенностью.

Обработка nodeid для составления отчета о тестировании происходит здесь:
https://a.yandex-team.ru/arc_vcs/library/python/pytest/yatest_tools.py?rev=a89a87076b2127a858f6c7e71b90894640957623#L282
"""

# mypy: allow-incomplete-defs, allow-untyped-defs, allow-untyped-calls

import warnings
from collections import defaultdict
from io import StringIO
from typing import Dict, List, Optional, Set, Union, cast

import py
from pytest import Class, Collector, File, Function, Item, Module

__all__ = (
    'pytest_addoption',
    'pytest_configure',
    'pytest_collection_modifyitems',
    'pytest_generate_tests',
)


class FunctionItem:
    """
    Вспомогательная обертка на Function для слежения за коллизиями по nodeid ее метафункции.
    Метафункция - это непосредственно функция с тестом, расположенная в модуле / классе,
    на основе которой генерируются элементы типа Function для каждой комбинации параматризованных фикстур.
    """

    def __init__(self, function_item: Function):
        self.item = function_item

        module_item: Optional[Module] = None
        class_items: List[Class] = []

        chain: List[Union[Item, Collector, File]] = cast(List[Union[Item, Collector, File]], function_item.listchain())
        # Function, Instance, Class, Instance, Class, ..., Module, Package, Session
        for item in reversed(chain):
            if isinstance(item, Class):
                class_items.append(item)
            elif isinstance(item, Module):
                module_item = item
                break
        assert module_item is not None
        class_items.reverse()
        self.module_item = module_item
        self.class_items = class_items

        self.metafunc_nodeid_id: str = self._get_metafunc_nodeid_id()
        self.arcadia_metafunc_nodeid_id: str = self._get_arcadia_nodeid()

    def _get_arcadia_nodeid(self) -> str:
        """
        Кривой аркадийный идентификатор метафункции, пропускающий промежуточные классы.
        """
        res = StringIO()
        res.write(self.module_item.nodeid)
        res.write('::')
        if self.class_items:
            root_class = self.class_items[0]
            res.write(root_class.name)
            res.write('::')
        res.write(self.item.obj.__name__)
        return res.getvalue()

    def _get_metafunc_nodeid_id(self) -> str:
        """
        Корректный идентификатор теста без суффикса с информацией о параметризации.
        С точки зрения pytest это называется метафункцией, на основе которой генерируются параметризованные вызовы.
        """
        res = StringIO()
        res.write(self.module_item.nodeid)
        res.write('::')
        for item in self.class_items:
            res.write(item.name)
            res.write('::')
        # nodeid может содержать суффикс параметризации для Function нод - нам нужно чистое имя функции
        res.write(self.item.obj.__name__)
        return res.getvalue()


class Registry:
    def __init__(self):
        self._registry: Dict[str, List['FunctionItem']] = defaultdict(list)
        self._duplicates: Set[str] = set()
        self._metafunc_nodeids: Set[str] = set()

    def get_metafuction_ids(self) -> List[str]:
        return sorted(self._metafunc_nodeids)

    def register(self, function_item: 'FunctionItem') -> None:
        if function_item.metafunc_nodeid_id in self._metafunc_nodeids:
            return

        self._metafunc_nodeids.add(function_item.metafunc_nodeid_id)

        self._registry[function_item.arcadia_metafunc_nodeid_id].append(function_item)
        if len(self._registry[function_item.arcadia_metafunc_nodeid_id]) > 1:
            self._duplicates.add(function_item.arcadia_metafunc_nodeid_id)

    def has_arcadia_nodeid_collision(self) -> bool:
        return bool(self._duplicates)

    def get_duplicated_items(self) -> Dict[str, List['FunctionItem']]:
        return {
            arcadia_nodeid: self._registry[arcadia_nodeid]
            for arcadia_nodeid in self._duplicates
        }


class Watcher:
    def __init__(self):
        self.registry = Registry()

    def register_metafunction(self, metafunc):
        item: Function = metafunc.definition
        self.registry.register(FunctionItem(item))

    def detect_conflicts(
        self,
        session,
        config,
        items: List[Item],
    ):
        if not self.registry.has_arcadia_nodeid_collision():
            return

        items.clear()

        warnings.warn('Collected tests have conflicting nodeids', RuntimeWarning)

        tw: py.io.TerminalWriter = config.get_terminal_writer()

        duplicates = self.registry.get_duplicated_items()
        arcadia_nodeids = sorted(duplicates.keys())
        for arcadia_nodeid in arcadia_nodeids:
            tw.write(f"{4 * ' '}Test items with the same arcadia nodeid ")
            tw.write(f"{arcadia_nodeid}\n", purple=True, bold=True)
            for function_item in sorted(duplicates[arcadia_nodeid], key=lambda item: item.metafunc_nodeid_id):
                tw.write(f"{8 * ' '}{function_item.metafunc_nodeid_id}\n", blue=True)

        raise RuntimeError('Collected tests have conflicting nodeids')

    def print_metafuncs(
        self,
        session,
        config,
        items: List[Item],
    ):
        tw: py.io.TerminalWriter = config.get_terminal_writer()
        tw.write('\n')
        tw.write('\n'.join(self.registry.get_metafuction_ids()), purple=True)
        tw.write('\n')


watcher = Watcher()


class PluginConfig:
    detect_conflicts = False
    print_metafunc_ids = False
    no_run = False


def pytest_addoption(parser):
    parser.addoption(
        "--detect-conflicts",
        action="store_true",
        dest="detect_conflicts",
        default=False,
        help="Detect test cases with same arcadia nodeid (which is different from normal pytest nodeid).",
    )
    parser.addoption(
        "--print-metafunc-ids",
        action="store_true",
        dest="print_metafunc_ids",
        default=False,
        help="Print node ids for all test metafunctions (without parametrization).",
    )
    parser.addoption(
        "--no-run",
        action="store_true",
        dest="no_run",
        default=False,
        help="Do not run tests",
    )


def pytest_configure(config):
    PluginConfig.detect_conflicts = bool(config.option.detect_conflicts)
    PluginConfig.print_metafunc_ids = bool(config.option.print_metafunc_ids)
    PluginConfig.no_run = bool(config.option.no_run)


def pytest_generate_tests(metafunc):
    if PluginConfig.print_metafunc_ids or PluginConfig.detect_conflicts:
        watcher.register_metafunction(metafunc)


def pytest_collection_modifyitems(session, config, items):
    if PluginConfig.print_metafunc_ids:
        watcher.print_metafuncs(session, config, items)
        if PluginConfig.no_run:
            items.clear()
        return

    if PluginConfig.detect_conflicts:
        watcher.detect_conflicts(session, config, items)
        if PluginConfig.no_run:
            items.clear()
        return
