from dataclasses import fields, is_dataclass
from typing import Any, Callable, TypeVar, Union, cast, overload

V = TypeVar('V')
T = TypeVar('T', bound=Callable[..., V])


@overload
def ensure_all_fields(dataclazz: T) -> T:
    pass


@overload
def ensure_all_fields(dataclazz: T, **kwargs: Any) -> V:
    pass


def ensure_all_fields(dataclazz: T, **kwargs: Any) -> Union[T, V]:
    """Конструктор датакласса, в котором все поля обязательные.

    Use case
    Частенько в датакласс добавляются новые поля.
    Представь себе ситуацию, что некоторые данные мапятся в датакласс A.
    При добавлении поля A.newfield можно забыть в каком-то месте в коде указать алгоритм маппинга данных
    в поле newfield.
    Чтобы такого не происходило, можно в тестах использовать функцию ensure_all_fields
    и пользоваться строгим сравнением.
    """
    def _constructor(**kwargs):
        assert is_dataclass(dataclazz)
        params = {}
        missing = []
        for field in fields(dataclazz):
            try:
                params[field.name] = kwargs[field.name]
            except KeyError:
                missing.append(field.name)

        if missing:
            raise KeyError(f'Missing fields: {missing}')
        return dataclazz(**params)
    if len(kwargs) == 0:
        return cast(T, _constructor)
    return _constructor(**kwargs)
