from typing import Any, Callable, Generic, Optional, TypeVar

from hamcrest.core.base_matcher import BaseMatcher
from hamcrest.core.description import Description
from hamcrest.core.helpers.wrap_matcher import wrap_matcher

T = TypeVar('T')


class AfterProcessingMatcher(BaseMatcher, Generic[T]):
    def __init__(self, processor: Callable[[Any], T], submatcher: BaseMatcher):
        self.processor = processor
        self.submatcher = submatcher

    def matches(self, item: Any, mismatch_description: Optional[Description] = None) -> bool:
        try:
            processed = self.processor(item)
        except Exception as e:
            if mismatch_description:
                mismatch_description.append_description_of(item)                          \
                                    .append_text(' failed to be processed by processor ') \
                                    .append_description_of(self.processor)                \
                                    .append_text(' with the exception ')                  \
                                    .append_description_of(e)
            return False
        else:
            if mismatch_description:
                mismatch_description.append_text('<')
                self._describe_processor(mismatch_description)
            result = self.submatcher.matches(processed, mismatch_description)
            if mismatch_description:
                mismatch_description.append_text('>')
            return result

    def describe_to(self, description: Description) -> None:
        self.submatcher.describe_to(description)
        description.append_text(' ')
        self._describe_processor(description)

    def describe_mismatch(self, item: Any, mismatch_description: Description) -> None:
        self.matches(item, mismatch_description)

    def _describe_processor(self, description: Description) -> None:
        description.append_text('after processing with <' + self.processor.__name__ + '> ')


def convert_then_match(processor: Callable[[Any], T], matcher: Any) -> AfterProcessingMatcher[T]:
    return AfterProcessingMatcher(processor, wrap_matcher(matcher))
