from dataclasses import fields, is_dataclass
from typing import Any, Mapping, Sequence

from hamcrest.core.core.isequal import IsEqual
from hamcrest.core.description import Description
from hamcrest.core.matcher import Matcher
from hamcrest.library.integration.match_equality import EqualityWrapper


class NoSuchAttribute:
    def __repr__(self):
        return '<No such attribute>'

    def __eq__(self, o):
        return False


class IsEqualPumped(IsEqual):
    def describe_mismatch(self, item: Any, mismatch_description: Description) -> None:
        if item is None:
            mismatch_description.append_text('was None')
            return

        if isinstance(item, Mapping) and isinstance(self.object, Mapping):
            success = self._try_describe_mappings_mismatch(item, mismatch_description)
            if success:
                return
        if is_dataclass(self.object) and is_dataclass(item):
            success = self._try_describe_dataclasses_mismatch(item, mismatch_description)
            if success:
                return
        sequence_types = (list, tuple)
        if isinstance(self.object, sequence_types) and isinstance(item, sequence_types):
            success = self._try_describe_sequences_mismatch(item, mismatch_description)
            if success:
                return

        super().describe_mismatch(item, mismatch_description)

    def _try_describe_mappings_mismatch(self, item: Mapping, mismatch_description: Description) -> bool:
        item_keys = item.keys()
        object_keys = self.object.keys()
        if (missing_keys := object_keys - item_keys):
            mismatch_description.append_value(item) \
                .append_text(' is missing keys ') \
                .append_value(missing_keys)
            return True
        if (extra_keys := item_keys - object_keys):
            mismatch_description.append_value(item) \
                .append_text(' has extra keys ') \
                .append_value(extra_keys)
            return True

        for key in item_keys:
            object_value = self.object[key]
            item_value = item[key]
            if item_value != object_value:
                mismatch_description.append_text('value of key ') \
                    .append_value(key) \
                    .append_text(' ')
                submismatch_descriptor = self._get_submismatch_descriptor(object_value)
                submismatch_descriptor.describe_mismatch(item_value, mismatch_description)
                return True

        return False

    def _try_describe_dataclasses_mismatch(self, item: Any, mismatch_description: Description) -> bool:
        assert is_dataclass(item)

        for field in fields(self.object):
            if not field.compare:
                continue
            object_value = getattr(self.object, field.name)
            item_value = getattr(item, field.name, NoSuchAttribute())
            if object_value != item_value:
                mismatch_description.append_text('value of property ') \
                    .append_value(field.name) \
                    .append_text(' ')
                submismatch_descriptor = self._get_submismatch_descriptor(object_value)
                submismatch_descriptor.describe_mismatch(item_value, mismatch_description)
                return True

        return False

    def _try_describe_sequences_mismatch(self, item: Sequence, mismatch_description: Description) -> bool:
        if len(self.object) != len(item):
            mismatch_description.append_text('sequence expected to be of size ') \
                .append_value(len(self.object)) \
                .append_text(' but actual sequence has size ') \
                .append_value(len(item))
            return True

        for idx, expected, actual in zip(range(len(self.object)), self.object, item):
            if expected != actual:
                mismatch_description.append_text('value at index ') \
                    .append_value(idx) \
                    .append_text(' ')
                submismatch_descriptor = self._get_submismatch_descriptor(expected)
                submismatch_descriptor.describe_mismatch(actual, mismatch_description)
                return True

        return False

    def _get_submismatch_descriptor(self, object_value):
        if isinstance(object_value, EqualityWrapper):
            object_value = object_value.matcher
        if isinstance(object_value, Matcher):
            return object_value
        return IsEqualPumped(object_value)


def equal_to(obj):
    """Matches if object is equal to a given object.

    :param obj: The object to compare against as the expected value.

    Базовый equal_to слабоват. Просто печатает expected: <object< but was: <actual>
    Ну что это такое? Нихрена же не понятно.
    Возможно, стоит законтрибьютить в https://github.com/hamcrest/PyHamcrest.
    """
    return IsEqualPumped(obj)
