import inspect
import unittest.mock
import warnings
from dataclasses import dataclass
from functools import partial
from typing import Any

import pytest

from sendr_pytest.matchers import equal_to

from hamcrest import assert_that, contains, instance_of, match_equality


class ActionMock:
    """
    Это мок для mock_action v2.0 с новыми удобными методами.

    sendr-qtools Action - кусок бизнес логики, который инстанцируется с помощью конструктора,
    а запускается с помощью метода run.
    Запустить экшен = вызвать конструктор с определенными аргументами и выполнить корутину run.
    Проверка, что экшен был вызван = AND(
        конструктор был вызван с конкретными аргументами
        выл вызван await на корутину run
    )
    Это требование нельзя выразить с помощью одного экземпляра unittest.mock.Mock. Нужно два.
    Собственно, класс ActionMock представляет новые методы: assert_run_with и assert_run_once_with.

    Мотивация:
    Использовать mock_action v1.0 - опасно. Допустим, ты вызываешь экшен авторизации. Тебя не волнует
    результат его работы. Тебе важно, чтобы он был вызван, и не упал (упал = авторизация провалена).

    При этом, он совместим с mock_action v1.0. Отнаследован от mock, поэтому если обращаться с ним
    как с mock_action v1.0, то получишь ожидаемое поведение: он, как и mock_action v1.0, ведёт себя как
    мок метода __init__ (при этом run тоже замокан но его нельзя проверить - как и в mock_action v1.0)

    В названиях метода используется run, а не ran, потому что тут необходима третья форма (past participle).
    Если я не прав, то поправьте плиз.
    """

    def __init__(self, init_mock, run_mock, run_async_mock):
        self._init_mock = init_mock
        self._run_mock = run_mock
        self._run_async_mock = run_async_mock

    def __getattr__(self, name):
        return getattr(self._init_mock, name)

    @property
    def action_return_value(self):
        return self._run_mock.return_value

    @property
    def side_effect(self):
        return self._run_mock.side_effect

    @side_effect.setter
    def side_effect(self, value):
        self._run_mock.side_effect = value

    def _get_child_mock(self, **kwargs):
        """
        https://docs.python.org/3/library/unittest.mock-examples.html#coping-with-mutable-arguments
        > When you subclass Mock or MagicMock all dynamically created attributes,
        > and the return_value will use your subclass automatically.
        > That means all children of a CopyingMock will also have the type CopyingMock.

        Когда делаешь mock.foo.bar, то foo и bar это тоже моки.
        У наследников Mock дочерние моки создаются по умному.
        Использовать ActionMock для дочерних моков - странно. Стоит использовать mocker.Mock.
        Но нижнее подчеркивание в названии _get_child_mock говорит мне о том, что это не часть API.
        Поэтому давайте аккуратно отключим дочерние моки, чтобы никто не завязался.
        """
        raise TypeError('Child mocks are not allowed')

    def assert_run_with(self, *args, **kwargs):
        self._init_mock.assert_called_with(*args, **kwargs)
        self._run_mock.assert_awaited_with()

    def assert_run(self):
        self._init_mock.assert_called()
        self._run_mock.assert_awaited_with()

    def assert_run_once_with(self, *args, **kwargs):
        self._init_mock.assert_called_once_with(*args, **kwargs)
        self._run_mock.assert_awaited_once_with()

    def assert_run_once(self):
        self._init_mock.assert_called_once()
        self._run_mock.assert_awaited_once_with()

    def assert_not_run(self):
        self._init_mock.assert_not_called()
        self._run_mock.assert_not_awaited()

    def assert_has_runs(self, calls, any_order=False):
        self._init_mock.assert_has_calls(calls, any_order=any_order)
        self._run_mock.assert_has_awaits([unittest.mock.call() for _ in range(len(calls))])


class ActionMockPatcher:
    def __init__(
        self,
        cls,
        return_value=unittest.mock.DEFAULT,
        side_effect=None,
        *,
        mock_cls=unittest.mock.Mock,
        async_mock_cls=unittest.mock.AsyncMock,
    ):
        if isinstance(return_value, Exception) or (
            inspect.isclass(return_value) and issubclass(return_value, Exception)
        ):
            warnings.warn(
                '`mock_action(action_cls, exc)` is a deprecated use of mock_action. '
                'Do `mock_action(action_cls, side_effect=exc) instead`'
            )
            side_effect = return_value
            return_value = unittest.mock.DEFAULT

        _init_mock = mock_cls(return_value=None)
        _run_mock = async_mock_cls(return_value=return_value, side_effect=side_effect)
        _run_async_mock = async_mock_cls()
        self._action_mock = ActionMock(_init_mock, _run_mock, _run_async_mock)
        self._subpatchers = []
        self._subpatchers.append(unittest.mock.patch.object(cls, '__init__', _init_mock))
        self._subpatchers.append(unittest.mock.patch.object(cls, 'run', _run_mock))
        if hasattr(cls, 'run_async'):
            self._subpatchers.append(unittest.mock.patch.object(cls, 'run_async', _run_async_mock))

    def start(self):
        for patcher in self._subpatchers:
            patcher.start()
        return self._action_mock

    def stop(self):
        """
        Останавливает патчинг экшена.
        """
        # Порядок вызова stop разных патчеров - важен. Нужно останавливать патчеры в обратном порядке
        # Иллюстрация проблемы:
        # Представь себе что мы мокаем один и тот же объект два раза
        # p1 = patch(obj, 'method', mock1)
        # p2 = mock(obj, 'method', mock2)

        # Если вызвать p1.stop(); p2.stop(), то на месте obj.method окажется mock1.
        # Логично же?
        # 1. Сначала p1.stop() убирает то, что есть на месте method и кладёт туда оригинальный method.
        # 2. Потом p2.stop() убирает то, что есть на месте method и кладёт туда то, что там было изначально.
        #    А изначально там был mock1.
        while self._subpatchers:
            patcher = self._subpatchers.pop()
            patcher.stop()


@pytest.fixture
def mock_action(mocker):
    """
    mock_action v2.0.
    """
    patchers = []

    def _mock_action(cls, *args, **kwargs):
        nonlocal patchers
        patcher = ActionMockPatcher(cls, *args, **kwargs, mock_cls=mocker.Mock, async_mock_cls=mocker.AsyncMock)
        patchers.append(patcher)
        mock = patcher.start()
        return mock

    yield _mock_action

    while patchers:
        patcher = patchers.pop()
        patcher.stop()


@pytest.fixture
def spy_action(mocker):
    class FixSpyClassAssert:
        """
        Если применить spy к bound методу, то assert_called будет работать неудобно.

        class A:
            def foo(self, x: int) -> int: return x * 2

        spy = mocker.spy(A, 'foo')
        A().foo(10)
        spy.assert_called_once_with(10)  # падает
        spy.assert_called_once_with(match_equality(instance_of(A)), 10)  # не падает

        Этот класс фиксит такое поведение, подразумевая что первый аргумент это всегда экземпляр класса
        """
        def __init__(self, cls, spy):
            self._cls = cls
            self._spy = spy

        def __getattr__(self, name):
            value = getattr(self._spy, name)
            if name.startswith('assert_') and name.endswith('_with') and not name.startswith('assert_not_'):
                return partial(value, match_equality(instance_of(self._cls)))
            return value

    class ActionSpy(ActionMock):
        def __init__(self, action_cls, mocker):
            self._init_spy = FixSpyClassAssert(action_cls, mocker.spy(action_cls, '__init__'))
            self._run_spy = FixSpyClassAssert(action_cls, mocker.spy(action_cls, 'run'))
            self._run_async_spy = None
            if hasattr(action_cls, 'run_async'):
                self._run_async_spy = FixSpyClassAssert(action_cls, mocker.spy(action_cls, 'run_async'))

            super().__init__(init_mock=self._init_spy, run_mock=self._run_spy, run_async_mock=self._run_async_spy)

    def _spy_action(cls):
        return ActionSpy(cls, mocker)

    return _spy_action


@dataclass
class call:
    args: Any
    kwargs: Any


@pytest.fixture(autouse=True)
def explain_call_asserts(mocker):
    """Фикстура пригодится, если хочешь разобраться в сложном упавшем assert_called_with etc.

    Хак на хаке. Тут и переопределение приватного метода фикстуры, и миксинами обмазано,
    и в mocker вмешиваемся, и какой-то непонятный датакласс используется.
    Но если ты просто устал, и хочешь просто увидеть, в чём, чёрт возьми, отличие - give it a shot.
    """
    class CustomFailureMessageMixin:
        def _format_mock_failure_message(self, args, kwargs, action='call'):
            if action in ('call', 'await'):
                actual_args, actual_kwargs = self.call_args
                try:
                    assert_that(
                        call(args=actual_args, kwargs=actual_kwargs),
                        equal_to(
                            call(
                                args=match_equality(contains(*args)),
                                kwargs=kwargs,
                            )
                        )
                    )
                except AssertionError as exc:
                    return 'expected %s not found.\n%s' % (action, str(exc))
            return super()._format_mock_failure_message(args=args, kwargs=kwargs, action=action)

    class Mock(CustomFailureMessageMixin, mocker.Mock):
        pass

    class AsyncMock(CustomFailureMessageMixin, mocker.AsyncMock):
        pass

    mocker.Mock = Mock
    mocker.AsyncMock = AsyncMock
