from __future__ import annotations

import logging
from typing import Any, Dict, Optional, Union

from sendr_qlog.logging.context import Context


class LoggerContext(logging.LoggerAdapter):
    """
    Предлагает стандартный интерфейс логгера с возможностью добавления контекстных значений,
    которые будут переданы в специальном атрибуте на LogRecord в момент логирования события
    для последующего форматирования события.
    """

    def __init__(self, logger: logging.Logger, context: Optional[Dict[str, Any]] = None):
        super().__init__(logger, {})
        self._context: Context = Context(initial=context)

    def context_push(self, **kwargs):
        self._context.push(**kwargs)

    def context_pop(self, *keys):
        self._context.pop(*keys)

    def get_context(self) -> Dict[str, Any]:
        return self._context.get_top_view()

    def __enter__(self):
        self._context.__enter__()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self._context.__exit__(exc_type, exc_val, exc_tb)
        return False

    def with_logger(self, logger: Union[logging.Logger, str]) -> LoggerContext:
        """
        Создать экземпляр LoggerContext,
        заменив приемник сообщений, но сохранив общий контекст,
        если имеет смысл переключиться на новый применик сообщений, сохранив контекст логирования.

        Управление общим контекстом будет происходить через любой из LoggerContext объектов:
        старый и новый, в этом смысле они оба будут эквивалентны.
        """

        if isinstance(logger, str):
            logger = logging.getLogger(logger)

        instance = type(self)(logger=logger)
        instance._context = self._context
        return instance

    def clone(self) -> LoggerContext:
        """
        Создать новый экземпляр LoggerContext с копией контекста.
        """
        instance = type(self)(logger=self.logger)
        instance._context = self._context.clone()
        return instance

    def process(self, msg, kwargs):
        extra = kwargs.setdefault('extra', {})
        # позднее LogRecord будет иметь атрибут _context, когда попадет в форматтер и/или хендлер
        extra['_context'] = self.get_context()
        return msg, kwargs
