from collections import defaultdict
from copy import deepcopy
from typing import Any, Dict, List, Optional


class Context:
    """
    Хранит списки значений для ключей, с возможностью разделения списка
    на логические отрезки, соответствующие входу в некоторый контекст.
    Подходящим словом для отрезка был бы фрейм (?) Со списокм значений для ключа
    можно работать как со стеком, в котором актуальное значение по ключу лежит на вершине,
    но всегда можно вернуться к предыдущему значнию, выкинув текущую вершину стека.
    Все списки ключей глобально делятся на фреймы, чтобы в рамках одного фрейма управлять ключами
    независимо от нижележащих значений: все значения одного фрейма могут быть удалены разом и
    значения, лежащие в предыдущем фрейме не могут быть удалены, если сверху лежит новый фрейм,
    т. е. управление стеком значений ключа происходит в пределах текущего фрейма.

    Пример:
    >>> ctx = Context()
    >>> ctx.push(a=1)
    >>> assert ctx.get_top_view() == {'a': 1}
    >>> ctx.push(a=2)
    >>> assert ctx.get_top_view() == {'a': 2}
    >>> ctx.pop('a')
    >>> assert ctx.get_top_view() == {'a': 1}
    >>> with ctx:
    >>>     ctx.push(a=3)
    >>>     >>> assert ctx.get_top_view() == {'a': 3}
    >>> assert ctx.get_top_view() == {'a': 1}

    Вид сверху (по последним элементам списков) дает текущее состояние контекста.

    текущее  |  a3 |  b2 |  c4 |
    =========|=====|=====|=====|
             |     |     |
             |     |     |  c4 |
    фрейм 2  |     |  b2 |  c3 |
    ---------|-----|-----|-----|
             |  a3 |     |     |
             |  a2 |     |  c2 |
    фрейм 1  |  a1 |  b1 |  c1 |
    ---------|-----|-----|-----|
    ключи    |  a  |  b  |  c  |

    """

    def __init__(self, initial: Optional[Dict[str, Any]] = None):
        self._values: Dict[str, List[Any]] = defaultdict(list)
        self._counters: List[Dict[str, int]] = [defaultdict(int)]
        if initial is not None:
            self.push(**initial)

    def push_frame(self):
        """Начать новый фрем значений"""
        self._counters.append(defaultdict(int))

    def pop_frame(self):
        """Убрать фрейм и все принадлежащие ему значения"""
        # базовый фрейм должен быть всегда
        if len(self._counters) == 1:
            return

        context = self._values
        counters = self._counters.pop()
        for key, values_in_frame in counters.items():
            values_total = len(context[key])
            values_left = values_total - values_in_frame
            if values_left > 0:
                context[key] = context[key][:values_left]
            else:
                del context[key]

    def push(self, **kwargs):
        """Положить значения не вершины стеков соответствующих ключей в рамках текущего фрейма."""
        values = self._values
        counters = self._counters[-1]
        for key, value in kwargs.items():
            values[key].append(value)
            counters[key] += 1

    def pop(self, *keys: str):
        """
        Безопасно удалить значений с вершин стеков укзаанных ключей в текущем фрейме.
        Если ключа нет в текущем фрейме, то со стеком значений ключа ничего не произойдет.
        """

        values = self._values
        counters = self._counters[-1]

        for key in keys:
            if key not in counters:
                continue

            values[key].pop()
            counters[key] -= 1

            if counters[key] == 0:
                del counters[key]

            if len(values[key]) == 0:
                del values[key]

    def get_top_view(self) -> Dict[str, Any]:
        """Вернуть словарь, где каждому ключу соответствует текущая вершина его стека значений."""
        return {key: values_list[-1] for key, values_list in self._values.items()}

    def __enter__(self):
        self.push_frame()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.pop_frame()
        return False

    def clone(self):
        instance = type(self)(initial={})
        instance._values = deepcopy(self._values)
        instance._counters = deepcopy(self._counters)
        return instance
