import logging
from datetime import datetime, timezone
from typing import Any, Dict, Optional


class BaseStructureFormatter(logging.Formatter):
    """
    Adaptation of ylog.format.QloudJsonFormatter for asyncio

    Configuration example
        LOGGING = {
            'version': 1,
            'disable_existing_loggers': False,
            'formatters': {
                'qloud': {
                    '()': 'sendr_webloop.utils.logging.QloudFormatter',
                },
            },
            'handlers': {
                'default': {
                    'level': 'DEBUG',
                    'formatter': 'qloud',
                    'stream': sys.stdout,
                    'class': 'logging.StreamHandler',
                },
                'access': {
                    'level': 'DEBUG',
                    'formatter': 'qloud',
                    'class': 'logging.StreamHandler',
                }
            },
            'loggers': {
                'aiohttp.access': {
                    'handlers':['access'],
                    'level': 'DEBUG',
                    'propagate': False
                }
            },
            'root': {
                'handlers': ['default'],
                'level': 'DEBUG'
            }
        }
        logging.config.dictConfig(LOGGING)
    """

    LOG_RECORD_USEFUL_FIELDS = ('funcName', 'lineno', 'name')

    def _serialize(self, data):
        raise NotImplementedError

    def format(self, record):
        record.message = record.getMessage()

        log_data = {
            'message': record.message,
            'level': record.levelname,
            'datetime': datetime.now(timezone.utc).isoformat()
        }

        if record.exc_info:
            exc = logging.Formatter.formatException(self, record.exc_info)
            log_data['stackTrace'] = exc

        fields = {}

        standard_fields = self._get_standard_fields(record)
        if standard_fields:
            fields['std'] = standard_fields

        context = self._get_context(record)
        if context is not None:
            fields['context'] = context

        if fields:
            log_data['@fields'] = fields

        return self._serialize(log_data)

    def _get_standard_fields(self, record):
        return {
            field: getattr(record, field)
            for field in self.LOG_RECORD_USEFUL_FIELDS
            if hasattr(record, field)
        }

    def _get_context(self, record) -> Optional[Dict[str, Any]]:
        return getattr(record, '_context', None)
