from datetime import datetime, timezone
from typing import Any, Dict, Optional

from sendr_qlog.logging.formatters.base import BaseStructureFormatter
from sendr_qlog.logging.formatters.utils import force_unicode


class LinesFormatter(BaseStructureFormatter):

    def _serialize(self, data):
        ctx = None
        std = None
        fields = data.get('@fields')
        if fields:
            ctx = fields.get('context', {})
            std = fields.get('std', {})
        trace = data.get('stackTrace')

        msg_parts = [
            ('name', std['name']),
            ('level', data.get('level', 'ERROR')),
            ('message', data.get('message', '')),
            ('funcName', std.get('funcName', '')),
            ('lineno', std.get('lineno', '')),
            ('datetime', datetime.now(timezone.utc).isoformat()),
        ]

        if ctx:
            for con_key in sorted(ctx.keys()):
                msg_parts.append((con_key, ctx[con_key]))

        if trace:
            trace_str = '\n{}\n'.format('\n'.join((
                '\t{}'.format(line)
                for line in trace.splitlines()
            )))
        else:
            trace_str = ''
        return '\t'.join(
            ('{}={}'.format(k, v) for k, v in msg_parts)
        ) + trace_str

    def _get_context(self, record) -> Optional[Dict[str, Any]]:
        context = super(LinesFormatter, self)._get_context(record)
        if context:
            context = {
                force_unicode(k): force_unicode(v)
                for k, v in context.items()
            }
        return context
