import json
import logging
import os
from functools import partial

from deprecated import deprecated

_VAR_TAGS = (
    ('dc', ('DEPLOY_NODE_DC', 'QLOUD_DATACENTER')),
    ('q_host', ('DEPLOY_POD_PERSISTENT_FQDN', 'QLOUD_HOSTNAME')),
    ('yp_host', ('DEPLOY_POD_PERSISTENT_FQDN',)),
    ('docker_image', ('QLOUD_DOCKER_IMAGE',)),  # В деплое нет никакого аналога. Даже номер ревизии спеки не достать
    ('box', ('DEPLOY_BOX_ID',)),
)


class ContextHandlerMixin:
    def _emit(self, record, **kwargs):
        if hasattr(record, '_context'):
            d = {k: v for k, v in vars(record).items() if not k.startswith('__')}
            d['context'] = json.dumps(record._context)
            record = logging.makeLogRecord(d)
        super()._emit(record, **kwargs)


def get_environment():
    if 'DEPLOY_PROJECT_ID' in os.environ:
        return '{project}.{stage}'.format(
            project=os.environ['DEPLOY_PROJECT_ID'],
            stage=os.environ.get('DEPLOY_STAGE_ID', 'unknown'),
        )

    return '{project}.{application}.{environment}'.format(
        project=os.environ.get('QLOUD_PROJECT', '_dev'),
        application=os.environ.get('QLOUD_APPLICATION', '_dev'),
        environment=os.environ.get('QLOUD_ENVIRONMENT', '_dev'),
    )


def get_component():
    return os.environ.get('DEPLOY_UNIT_ID', os.environ.get('QLOUD_COMPONENT', '_dev'))


def get_instance():
    return os.environ.get('DEPLOY_POD_ID', os.environ.get('QLOUD_INSTANCE', '_dev'))


def get_tags():
    return {
        tag: os.environ[var] for tag, var_list in _VAR_TAGS for var in var_list
        if var in os.environ
    }


@deprecated
def raven_sentry_init(dsn,
                      *,
                      timeout=10,
                      transport=None,
                      component=None,
                      instance=None,
                      environment=None,
                      release=None,
                      ):
    """
    Функция настраивает отправку ошибок в sentry.
    Возвращается корутина, которую надо вызвать при завершении приложения.
    """
    try:
        import raven
        import raven_aiohttp
        from raven.conf import setup_logging
        from raven.handlers.logging import SentryHandler as BaseSentryHandler
    except ImportError:
        async def dummy(*args, **kwargs):
            pass
        return dummy

    class SentryHandler(ContextHandlerMixin, BaseSentryHandler):
        pass

    if transport is None:
        transport = partial(raven_aiohttp.AioHttpTransport, family=0)

    if not environment:
        environment = get_environment()

    if not component:
        component = get_component()

    if not instance:
        instance = get_instance()

    tags = get_tags()

    client = raven.Client(
        dsn,
        transport=transport,
        timeout=timeout,
        enable_breadcrumbs=False,
        environment=environment,
        name=component,
        site=instance,
        tags=tags,
        context={},
        release=release,
    )
    handler = SentryHandler(client)
    handler.setLevel(logging.ERROR)
    setup_logging(handler)

    async def close(*args, **kwargs):
        await client.remote.get_transport().close()

    return close


def sentry_sdk_sentry_init(dsn, *, release=None):
    import sentry_sdk
    from sentry_sdk.integrations.aiohttp import AioHttpIntegration
    from sentry_sdk.integrations.logging import EventHandler as BaseEventHandler
    from sentry_sdk.integrations.logging import LoggingIntegration as BaseLoggingIntegration

    class EventHandler(ContextHandlerMixin, BaseEventHandler):
        pass

    class LoggingIntegration(BaseLoggingIntegration):
        def __init__(self, level=logging.INFO, event_level=logging.ERROR):
            super().__init__(level=level, event_level=event_level)
            self._handler = EventHandler(level=event_level)

    sentry_sdk.init(
        dsn=dsn,
        default_integrations=False,
        integrations=[LoggingIntegration(), AioHttpIntegration()],
        environment=get_environment(),
        release=release,
    )
    with sentry_sdk.configure_scope() as scope:
        scope.set_tag('component', get_component())
        scope.set_tag('instance', get_instance())
        for key, value in get_tags().items():
            scope.set_tag(key, value)


def sentry_init(*args, **kwargs):
    try:
        import sentry_sdk  # noqa
        return sentry_sdk_sentry_init(*args, **kwargs)
    except ImportError:
        pass

    try:
        import raven  # noqa
        return raven_sentry_init(*args, **kwargs)
    except ImportError:
        pass

    async def dummy(*args, **kwargs):
        pass
    return dummy
