import contextlib
import time
from typing import Callable

import ujson

from ... import Counter, Gauge, Histogram, registry
from ...registry import MetricsRegistry


def get_registry_handler(handle_registry=registry.REGISTRY):
    """
        Генерация хендлера для отдачи показателей статистики
    """
    from aiohttp import web

    async def _handler(request):
        return web.json_response(data=list(handle_registry.collect()), dumps=ujson.dumps)

    return _handler


def get_stats_middleware(handle_registry: MetricsRegistry = registry.REGISTRY) -> Callable:
    from aiohttp import web

    REQUEST_TIME = Histogram(
        'request_time',
        labelnames=('route_name', 'method'),
        registry=handle_registry,
    )

    # timings of handled requests - handler returned response object
    REQUEST_HANDLE_TIME = Histogram(
        'request_handle_time',
        labelnames=('route_name', 'method'),
        registry=handle_registry,
    )

    # counter of handled requests - handler returned response
    REQUEST_STATUS_COUNT = Counter(
        'request_status',
        labelnames=('route_name', 'method', 'status'),
        registry=handle_registry,
    )

    # number of requests being handled by application at the moment
    REQUEST_COUNT_GAUGE = Gauge(
        name='request_count',
        labelnames=('route_name', 'method'),
        registry=handle_registry,
    )

    @web.middleware
    async def middleware_stats(request, handler):
        """
        Middleware для автоматической инструментации обработки запросов
        """

        route = request.match_info.route
        route_name = route.name or 'unknown'
        method = request.method

        if route_name in ['stats', 'unistat']:
            return await handler(request)

        with contextlib.ExitStack() as ctx:
            ctx.enter_context(REQUEST_TIME.labels(route_name, method).time)
            ctx.enter_context(REQUEST_COUNT_GAUGE.labels(route_name, method).track)

            start = time.perf_counter()

            response = await handler(request)

            REQUEST_HANDLE_TIME.labels(route_name, method).observe(time.perf_counter() - start)

            status = str(response.status)
            generic_status = '{}xx'.format(status[0])
            REQUEST_STATUS_COUNT.labels(route_name, method, status).inc()
            REQUEST_STATUS_COUNT.labels(route_name, method, generic_status).inc()

        return response

    return middleware_stats
