import time
from typing import Awaitable, Callable, Optional

from aiohttp import web
from aiohttp.web_exceptions import HTTPException
from library.python.monlib.encoder import dumps
from library.python.monlib.metric_registry import HistogramType, MetricRegistry

CONTENT_TYPE_SPACK = 'application/x-solomon-spack'
CONTENT_TYPE_JSON = 'application/json'

HandlerType = Callable[[web.Request], Awaitable[web.Response]]


def get_registry_handler(registry: MetricRegistry) -> HandlerType:
    """
    Генерация хендлера для отдачи показателей статистики
    """
    from aiohttp import web

    async def _handler(request: web.Request) -> web.Response:
        if request.content_type == CONTENT_TYPE_SPACK or CONTENT_TYPE_SPACK in request.headers.get('Accept', ''):
            return web.Response(body=dumps(registry), content_type=CONTENT_TYPE_SPACK)
        return web.Response(body=dumps(registry, format='json'), content_type=CONTENT_TYPE_JSON)

    return _handler


def get_stats_middleware(registry: MetricRegistry) -> Callable[[web.Request, HandlerType], Awaitable[web.Response]]:
    @web.middleware
    async def middleware_stats(request: web.Request, handler: HandlerType) -> web.Response:
        """
        Middleware для автоматической инструментации обработки запросов
        """

        route = request.match_info.route
        route_name = route.name or 'unknown'
        method = request.method

        if route_name in ['stats', 'unistat', 'solomon']:
            return await handler(request)

        exception: Optional[Exception] = None
        status = None
        status_group = 'other'

        start = time.perf_counter()
        try:
            response = await handler(request)
            status = response.status
        except HTTPException as e:
            exception = e
            status = e.status
        except Exception as e:
            exception = e

        if status and len(f'{status}') == 3:
            status_group = f'{str(status)[0]}xx'

        labels = {
            'route': route_name,
            'method': method,
            'status': f'{status}',
            'statusGroup': status_group,
        }

        registry.rate({**labels, 'name': 'responses'}).inc()
        registry.histogram_rate({**labels, 'name': 'responsesTime'}, HistogramType.Explicit, buckets=[
            0, 5, 10, 25, 50, 75, 100, 250, 500, 750, 1000,
            2500, 5000, 7500, 10000, 12500, 15000, 17500, 20000, 25000, 30000, 60000
        ]).collect((time.perf_counter() - start) * 1000)

        if exception:
            raise exception
        return response  # noqa

    return middleware_stats
