import re
from threading import Lock

from .registry import REGISTRY

_METRIC_NAME_RE = re.compile(r'^[a-zA-Z_][a-zA-Z0-9_./@]*$')
_METRIC_LABEL_NAME_RE = re.compile(r'^[a-zA-Z][a-zA-Z0-9_./@]*$')


class _LabelWrapper(object):
    '''Handles labels for the wrapped metric.'''

    def __init__(self, wrappedClass, name, labelnames, **kwargs):
        self._wrappedClass = wrappedClass
        self._name = name
        self._labelnames = labelnames
        self._full_name = self._name + '_' + '_'.join((str(label) for label in labelnames))
        self._kwargs = kwargs
        self._lock = Lock()
        self._metrics = {}

        for label in labelnames:
            if label.startswith('__'):
                raise ValueError('Invalid label metric name: ' + label)

    def labels(self, *labelvalues):
        if len(labelvalues) != len(self._labelnames):
            raise ValueError('Incorrect label count')
        labels_suff = '_'.join((str(label) for label in labelvalues))
        with self._lock:
            if labels_suff not in self._metrics:
                self._metrics[labels_suff] = self._wrappedClass('{}_{}'.format(self._name, labels_suff), **self._kwargs)
            return self._metrics[labels_suff]

    def remove(self, *labelvalues):
        '''Remove the given labelset from the metric.'''
        if len(labelvalues) != len(self._labelnames):
            raise ValueError('Incorrect label count')
        labels_suff = '_'.join((str(label) for label in labelvalues))
        if labels_suff in self._metrics:
            with self._lock:
                del self._metrics[labels_suff]

    def get(self):
        with self._lock:
            metrics = tuple(self._metrics.values())
        for metric in metrics:
            for name, value in metric.get():
                yield (name, value)

    @property
    def full_name(self):
        return self._full_name


def _MetricWrapper(cls):
    def init(name, *, labelnames=(), registry=REGISTRY, **kwargs):
        if labelnames:
            labelnames = tuple(labelnames)
            for label in labelnames:
                if not _METRIC_LABEL_NAME_RE.match(label):
                    raise ValueError('Invalid label metric name: ' + label)
            metric = _LabelWrapper(cls, name, labelnames, **kwargs)
        else:
            metric = cls(name, **kwargs)

        if not _METRIC_NAME_RE.match(name):
            raise ValueError('Invalid metric name: ' + name)

        if registry:
            registry.register(metric)
        return metric

    init.__wrapped__ = cls
    return init
