"""
Адаптация сборщиков метрик из https://github.com/prometheus/client_python,
для работы с YASM форматом.

Классы метрик обернуты в фабричные функции-конструкторы.

Конструктор создает сборщик метрики либо группирующий объект,
если передать в labelnames непустой список строк.

Группирующий объект нужен, чтобы агрегировать/измерять метрики одного типа,
если каждое значение/измерение может быть аннотировано фиксированным набором атрибутов,
и имеет смысл учитывать/объединять измерения только с совпадающими атрибутами:
измерения с совпадающими значениями атрибутов идут в один и тот же сборщик метрики,
а отличным значениям атрибутов соответствуют разные сборщики метрики.

В группирующем объекте labelnames задает названия для атрибутов.
Позднее, по заданной n-ке значений атрибутов будет лениво создаваться сборщик метрики,
в котором следует учесть измерение.

Пример.
Нужно считать количество ответов некоторого сервиса с точностью до HTTP статус-кода,
(т. е. каждое измерение аннотировано значением статус кода, которому оно соответствует)
то нужно создать группирующий объект
    >>> count_http_codes Counter(labels=('http_status_code'))
и вносить измерения, указывая к какому статус-коду оно относится:
    >>> count_http_codes.labels(200).inc()
    >>> count_http_codes.labels(503).inc()
"""

import asyncio
import enum
from abc import ABCMeta, abstractmethod
from threading import Lock
from timeit import default_timer

from .labels import _MetricWrapper


class StrEnum(enum.Enum):
    def __str__(self):
        return self.value


class IntSuff(StrEnum):
    absolute = 'a'
    delta = 'd'


class AggSuff(StrEnum):
    hgram = 'h'
    summ = 'm'
    summnone = 'e'
    trnsp = 't'
    max = 'x'
    min = 'n'
    aver = 'v'
    counter = 'c'


class MetricSuffix(object):
    def __init__(self, interpret, group, metagroup, time):
        self._suff = '_{}{}{}{}'.format(interpret, group, metagroup, time)

    def __str__(self):
        return self._suff

    def __repr__(self):
        return str(self)


SUMM_SUFF = '_summ'
HGRAM_SUFF = '_hgram'
MAX_SUFF = '_max'
ABS_SUM_SUFF = MetricSuffix(IntSuff.absolute, AggSuff.summ, AggSuff.summ, AggSuff.trnsp)

DEFAULT_BUCKETS = (0.0, .005, .01, .025, .05, .075, .1, .25, .5, .75, 1.0, 2.5, 5.0, 7.5, 10.0)


class Metric(metaclass=ABCMeta):
    def __init__(self, name, suff, **kwargs):
        self._name = name
        self._suff = str(suff)

    @abstractmethod
    def get(self):
        return ()

    @property
    def name(self):
        return self._name

    @property
    def full_name(self):
        return self._name + self._suff


class _MutexValue(object):
    '''A float protected by a mutex.'''

    def __init__(self, *args, **kwargs):
        self._value = 0.0
        self._lock = Lock()

    def inc(self, amount):
        with self._lock:
            self._value += amount

    def set(self, value):
        with self._lock:
            self._value = value

    def get(self):
        with self._lock:
            return self._value


class SingleValueMetric(Metric):
    def __init__(self, name, suff, value=0.0, **kwargs):
        super().__init__(name, suff, **kwargs)
        self._value = _MutexValue()
        if value:
            self._value.set(value)

    def get(self):
        return ((self.full_name, self._value.get()),)


@_MetricWrapper
class Counter(SingleValueMetric):
    def __init__(self, name, suff=SUMM_SUFF, **kwargs):
        super().__init__(name, suff, **kwargs)

    def inc(self, amount=1.0):
        if amount < 0:
            raise ValueError('Increment Counter object by negative value is not allowed')
        self._value.inc(amount)

    def count_exceptions(self, exception=Exception):
        return _ExceptionCounter(self, exception)


class _WithTime:
    @property
    def time(self):
        return _TimeObserver(self)


@_MetricWrapper
class Gauge(SingleValueMetric, _WithTime):
    def __init__(self, name, suff=ABS_SUM_SUFF, **kwargs):
        super().__init__(name, suff, **kwargs)

    def inc(self, amount=1.0):
        self._value.inc(amount)

    def dec(self, amount=1.0):
        self._value.inc(-amount)

    def set(self, value):
        self._value.set(value)

    def observe(self, amount):
        self.set(amount)

    @property
    def track(self):
        return _InprogressTracker(self)


@_MetricWrapper
class Summary(Metric, _WithTime):
    def __init__(self, name, suff=SUMM_SUFF, **kwargs):
        super().__init__(name, suff)

        self._count = _MutexValue()
        self._sum = _MutexValue()

    def observe(self, amount):
        self._count.inc(1)
        self._sum.inc(amount)

    def get(self):
        return (
            ('{name}_sum{suff}'.format(name=self._name, suff=self._suff), self._sum.get()),
            ('{name}_count{suff}'.format(name=self._name, suff=self._suff), self._count.get()),
        )


@_MetricWrapper
class Histogram(Metric, _WithTime):
    def __init__(self, name, suff=SUMM_SUFF, buckets=DEFAULT_BUCKETS, **kwargs):
        super().__init__(name, SUMM_SUFF, **kwargs)
        buckets = [float(b) for b in buckets]
        if buckets != sorted(buckets):
            raise ValueError('Buckets not in sorted order')
        if len(buckets) < 2:
            raise ValueError('Must have at least two buckets')
        self._upper_bounds = tuple(reversed(buckets))
        self._buckets = [_MutexValue() for x in buckets]
        self._sum = _MutexValue()

    def observe(self, amount):
        self._sum.inc(amount)
        for i, bound in enumerate(self._upper_bounds):
            if amount >= bound:
                self._buckets[i].inc(1)
                break

    def get(self):
        return (
            ('{name}_sum{suff}'.format(name=self._name, suff=self._suff), self._sum.get()),
            (self.name + HGRAM_SUFF, tuple(
                ((x[0], x[1].get()) for x in zip(reversed(self._upper_bounds), reversed(self._buckets)))
            ))
        )


class _ContextWrapper(object):
    def __call__(self, func):
        if asyncio.iscoroutinefunction(func):
            async def wrapped(*args, **kwargs):
                with self:
                    return await func(*args, **kwargs)
        else:
            def wrapped(*args, **kwargs):
                with self:
                    return func(*args, **kwargs)
        return wrapped


class _TimeObserver(_ContextWrapper):
    def __init__(self, metric):
        self._metric = metric

    def __enter__(self):
        self._start = default_timer()

    def __exit__(self, typ, value, traceback):
        # Time can go backwards.
        self._metric.observe(max(default_timer() - self._start, 0))


class _ExceptionCounter(_ContextWrapper):
    def __init__(self, counter, exception):
        self._counter = counter
        self._exception = exception

    def __enter__(self):
        pass

    def __exit__(self, exc_type, exc_val, exc_tb):
        if isinstance(exc_val, self._exception):
            self._counter.inc()
        return False


class _InprogressTracker(_ContextWrapper):
    def __init__(self, gauge):
        self._gauge = gauge

    def __enter__(self):
        self._gauge.inc()

    def __exit__(self, typ, value, traceback):
        self._gauge.dec()
