"""
    Модуль настроек
"""

import glob
import logging
import os
from collections.abc import MutableMapping
from typing import Iterator, Optional

import yenv

from sendr_aiopg import EngineUnion
from sendr_qlog.logging.adapters.logger import LoggerContext
from sendr_qstats import MetricsRegistry
from sendr_settings.exceptions import HotSettingsUnsupportedTypeException
from sendr_settings.hot import AbstractSource, HotSetting, PgSource, SupportedTypes
from sendr_settings.stats import init_hot_settings_using_failures, safe_inc_using_failures

default_logger = logging.getLogger(__name__)


class Readable:
    def __init__(self, path: str):
        self.path: str = path

    def read(self) -> str:
        with open(self.path) as f:
            return f.read()


DefaultReadable = OverwriteReadable = Readable

try:
    from library.python import resource

    class ResourceReadable(Readable):
        def read(self):
            return resource.find(self.path).decode('utf-8')

    DefaultReadable = ResourceReadable

    def list_files(pathname):
        return list(resource.iterkeys(pathname))

except ImportError:
    def list_files(pathname):
        return glob.glob(pathname.rstrip('/') + '/*')


class Config(MutableMapping):
    _settings: dict
    _fields: set = {
        '_fields',
        '_settings',
        'hot_source',
        'hot_settings_inited',
        'hot_settings_using_failures',
        'logger',
    }

    def _update_hot_setting(self, key, setting):
        if type(setting.fallback_value) not in SupportedTypes:
            raise HotSettingsUnsupportedTypeException
        if self.hot_settings_inited:
            self.hot_source.set(key, setting)

    def _update_setting(self, key, value):
        if isinstance(value, HotSetting):
            self._update_hot_setting(key, value)
        self._settings[key] = value

    def _get_hot_setting(self, key, setting):
        if not self.hot_settings_inited:
            safe_inc_using_failures()
            self.logger.error('Unexpected getting uninitialized hot settings')
            return setting.fallback_value
        return self.hot_source.get(key)

    def _get_setting(self, key):
        value = self._settings[key]
        if isinstance(value, HotSetting):
            return self._get_hot_setting(key, value)
        return value

    def __init__(self, defaults=None):
        self._settings = defaults or dict()
        self.hot_source: Optional[AbstractSource] = None
        self.hot_settings_inited = False
        self.hot_settings_using_failures = None
        self.logger = default_logger

    def __getitem__(self, item):
        return self._get_setting(item)

    def __setitem__(self, key, value):
        self._update_setting(key, value)

    def __delitem__(self, key):
        del self._settings[key]

    def __iter__(self):
        return iter(self._settings)

    def __len__(self):
        return len(self._settings)

    def __getattr__(self, item):
        try:
            return self._get_setting(item)
        except KeyError:
            raise AttributeError('Setting %s does not exist' % item)

    def __setattr__(self, item, value):
        if item in self._fields:
            super().__setattr__(item, value)
        else:
            self._update_setting(item, value)

    def __delattr__(self, item):
        try:
            del self[item]
        except KeyError:
            raise AttributeError('Setting %s does not exist' % item)

    async def _init_hot_settings_values(self):
        for key, value in self._settings.items():
            if isinstance(value, HotSetting):
                self.hot_source.set(key, value)
        await self.hot_source.initialize()

    async def add_pg_hot_source(
        self,
        db_engine: EngineUnion,
        logger: LoggerContext,
        metrics_registry: Optional[MetricsRegistry] = None,
    ) -> None:
        self.logger = logger
        self.hot_source = PgSource(db_engine=db_engine, logger=logger, metrics_registry=metrics_registry)
        self.hot_settings_inited = True
        if metrics_registry is not None:
            init_hot_settings_using_failures(metrics_registry)
        await self._init_hot_settings_values()

    async def close_hot_source(self):
        if self.hot_source:
            await self.hot_source.close()

    @classmethod
    def get_config_files(cls, settings_dir: str, env: str, overwrite_file: Optional[str] = None) -> Iterator[Readable]:
        """
            Сбор файлов для конфигруации
            :param settings_dir: директория с файлами конфигурации
            :param env: суффикс окружения для выбора конфигов, различных в разных окружениях
            :param overwrite_file: файл локальных настроек
        """
        configs = sorted([
            path
            for path in list_files(settings_dir)
            if path.endswith('.conf') or path.endswith('.conf.{}'.format(env))
        ])

        for config in configs:
            yield DefaultReadable(config)

        if overwrite_file and os.path.exists(overwrite_file):
            yield OverwriteReadable(overwrite_file)

    @classmethod
    def load_from_env(cls, settings_dir, env=yenv.type, overwrite_file=None):
        config = cls()
        for conf in cls.get_config_files(settings_dir, env, overwrite_file=overwrite_file):
            ns = {}
            code = compile(conf.read(), conf.path, 'exec')
            eval(code, ns)

            for key, val in ns.items():
                if key.isupper():
                    config[key] = val

        return config
