from dataclasses import dataclass, field
from datetime import datetime
from typing import Dict, Optional

import sqlalchemy as sa

from sendr_aiopg import BaseMapperCRUD, CRUDQueries, Entity, StorageAnnotatedMeta, StorageBase, StorageContextBase
from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_utils import alist, utcnow

metadata = sa.MetaData(schema='public')

t_settings = sa.Table(
    'settings', metadata,
    sa.Column('key', sa.Text(), primary_key=True, nullable=False),
    sa.Column('value', sa.Text(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)


@dataclass
class SettingsEntity(Entity):
    key: str
    value: str
    created: Optional[datetime] = field(default_factory=utcnow)


class SettingsDataMapper(SelectableDataMapper):
    entity_class = SettingsEntity
    selectable = t_settings


class SettingsDataDumper(TableDataDumper):
    entity_class = SettingsEntity
    table = t_settings


class SettingsMapper(BaseMapperCRUD[SettingsEntity]):
    model = SettingsEntity

    _builder = CRUDQueries(
        base=t_settings,
        id_fields=('key',),
        mapper_cls=SettingsDataMapper,
        dumper_cls=SettingsDataDumper,
    )

    async def get_all_settings(self) -> Dict[str, str]:
        return {setting.key: setting.value for setting in await alist(self.find())}


class SettingsStorage(StorageBase, metaclass=StorageAnnotatedMeta):
    settings: SettingsMapper


class SettingsStorageContext(StorageContextBase):
    STORAGE_CLS = SettingsStorage
