import asyncio
from asyncio import Task
from typing import ClassVar, Dict, Optional, Union

from sendr_aiopg import EngineUnion
from sendr_qlog.logging.adapters.logger import LoggerContext
from sendr_qstats import MetricsRegistry
from sendr_settings.db import SettingsStorageContext
from sendr_settings.exceptions import HotSettingsUnsupportedTypeException
from sendr_settings.stats import init_hot_settings_refresh_failures, safe_inc_refresh_failures
from sendr_utils.abc import abstractclass

SupportedTypes = (float, int, str, bool)
SettingType = Union[float, int, str, bool]


class HotSetting:
    """
        Горячие настройки

        Обновляются раз в 30 секунд

        Как использовать:
        Добавить миграцию с таблицей settings в схему public
        Включить поддержку на объекте Conf(), вызвав add_pg_hot_settings при старте приложения
        Закрыть при остановке приложения, вызвав close_hot_source

        Пример:
            async def add_hot_settings(self, _: Any) -> None:
                await settings.add_pg_hot_source(
                    db_engine=self.db_engine,
                    logger=self.logger,
                    metrics_registry=REGISTRY, # нужно передавать для записи метрик
                )

            async def close_hot_settings(self, _: Any) -> None:
                await settings.close_hot_source()

            settings = Conf.load_from_env(...)
            self.on_startup.append(self.add_hot_settings)
            self.on_cleanup.append(self.close_hot_settings)

        Убедитесь, что pg_pinger проинициализирован до подключения горячих настроек. Можно так:
            self.on_startup.append(self.wait_for_pg_pinger_init)
            self.on_startup.append(self.add_hot_settings)

            async def wait_for_pg_pinger_init(self, _: Any) -> None:
                for i in range(10):
                    try:
                        async with StorageContext(db_engine=self.db_engine, logger=self.logger) as storage:
                            await storage.conn.execute('select 1;')
                            return
                    except TypeError:
                        await asyncio.sleep(1)

                raise RuntimeError('pg_pinger wait attempts have expired')

        Пример присваивания настроек:
            from sendr_settings import HotSetting
            API_CORS_ENABLED = HotSetting(fallback_value=True)

        fallback_value - значение, которое будет использовано, если не удастся достучаться до источника. Или
        он не добавлен. По сути fallback_value может быть применен только в двух случаях: лежит база, настройки
        некорректны или отсутствуют - это некоторый способ дать возможность работать сервису с деградацией

        Тип настройки выводится из fallback_value. После получения значения из источника оно будет приведено к
        соответствующему типу

        Поддерживаемые типы:
            string
            int
            bool
            float

        Рекомендуется настроить алерты по метрикам из файла stats.py
    """

    def __init__(self, fallback_value: SettingType):
        self.fallback_value = fallback_value


class CacheItem:
    def __init__(self, key: str, hot_setting: HotSetting):
        self.key = key
        self.value = hot_setting.fallback_value
        self.setting_type = type(hot_setting.fallback_value)


@abstractclass
class AbstractSource:
    def set(self, key: str, setting: HotSetting) -> None:
        raise NotImplementedError

    def get(self, key: str) -> SettingType:
        raise NotImplementedError

    async def initialize(self) -> None:
        raise NotImplementedError

    async def close(self) -> None:
        raise NotImplementedError


class PgSource(AbstractSource):
    refresh_interval: ClassVar[float] = 30.0

    def __init__(
        self,
        db_engine: EngineUnion,
        logger: LoggerContext,
        metrics_registry: Optional[MetricsRegistry] = None,
    ):
        self.db_engine = db_engine
        self.logger = logger
        self.cache: Dict[str, CacheItem] = dict()
        self.refresh_task: Optional[Task] = None
        self.wait_task: Optional[Task] = None
        self.is_running = False
        if metrics_registry is not None:
            init_hot_settings_refresh_failures(metrics_registry)

    def _map_bool(self, value: str) -> bool:
        if value in ['False', 'false', '0']:
            return False
        if value in ['True', 'true', '1']:
            return True

        self.logger.warning(f'Unrecognized bool setting value. Will fallback to True. Raw value: {value}')
        return True

    def _map_to_expected_type(self, item: CacheItem, db_value: str) -> SettingType:
        if item.setting_type == str:
            return db_value
        if item.setting_type == bool:
            return self._map_bool(db_value)
        if item.setting_type == int:
            return int(db_value)
        if item.setting_type == float:
            return float(db_value)
        raise HotSettingsUnsupportedTypeException

    async def _refresh_setting_from_db(self) -> None:
        try:
            if len(self.cache) == 0:  # Zero hot settings count
                return

            self.logger.info('Begin refreshing settings')
            async with SettingsStorageContext(db_engine=self.db_engine, logger=self.logger) as storage:
                settings = await storage.settings.get_all_settings()
            for key, item in self.cache.items():
                try:
                    self.logger.info(f'Update setting {key} with raw value {settings[key]}')
                    item.value = self._map_to_expected_type(item, settings[key])
                except Exception:
                    safe_inc_refresh_failures()
                    self.logger.exception(f'An error occurred on refreshing the {item.key} setting')
        except Exception:
            safe_inc_refresh_failures()
            self.logger.exception('An error occurred on getting hot settings from database')

    async def _refresh_settings_loop(self) -> None:
        while self.is_running:
            self.wait_task = asyncio.create_task(asyncio.sleep(self.refresh_interval))
            try:
                await self.wait_task
            except asyncio.CancelledError:
                return
            await self._refresh_setting_from_db()

    def set(self, key: str, setting: HotSetting) -> None:
        self.cache[key] = CacheItem(key, setting)

    def get(self, key: str) -> SettingType:
        item = self.cache[key]
        return item.value

    async def initialize(self):
        await self._refresh_setting_from_db()
        self.is_running = True
        self.refresh_task = asyncio.create_task(self._refresh_settings_loop())

    async def close(self):
        if not self.refresh_task:
            return

        self.is_running = False
        if self.wait_task:
            self.wait_task.cancel()
            self.wait_task = None

        try:
            await self.refresh_task
        except asyncio.CancelledError:
            pass
        finally:
            self.refresh_task = None
