import asyncio
from abc import ABCMeta, abstractmethod
from typing import Any, Callable, ClassVar, Dict, Iterable, Optional, Tuple, Type

from aiohttp import web

from sendr_aiohttp import BaseUrlDispatcher, Url
from sendr_aiohttp.handler import BaseHandler
from sendr_aiopg.types import EngineUnion
from sendr_qlog import LoggerContext
from sendr_taskqueue.background import BackgroundSchedule
from sendr_taskqueue.logger import default_logger
from sendr_taskqueue.stats import StatsTask
from sendr_taskqueue.worker.base import BaseWorker
from sendr_taskqueue.worker.base.arbiter import BaseArbiterWorker
from sendr_taskqueue.worker.base.monitoring.actions import get_check_handler
from sendr_taskqueue.worker.storage.monitoring.actions import WorkersHealthCheckAction


class BaseWorkerApplication(web.Application):
    """
    Класс для запуска сервера  отдачи статистики очереди обработки
    В наследнике надо переопределить:
        workers  парами [Класс воркера, количество запускаемых одновременно]
        arbiter_cls классом с реализацией обслуживания очереди
    Все дополнительные конфигурации и глобальный стейт можно сложить в атрибуты
    инстанса этого класса - он доступен в воркерах
    """
    routes: Iterable[Url] = ()
    middlewares: ClassVar[Tuple[Callable]] = ()  # type: ignore
    workers: ClassVar[Iterable[Tuple[Type[BaseWorker], int]]] = []  # Pairs of (<worker_class>, <instance count>)
    arbiter_cls: ClassVar[Type[BaseArbiterWorker]]  # Arbiter class (cleanup, watch for tasks, etc)
    debug: ClassVar[bool] = False
    sentry_dsn: ClassVar[Optional[str]] = None
    version: ClassVar[Optional[str]] = None
    logger_ctx_cls: ClassVar[Type[LoggerContext]] = LoggerContext

    def __init__(self) -> None:
        super().__init__(router=BaseUrlDispatcher(), middlewares=self.middlewares, debug=self.debug)
        self.on_startup.append(self.setup)

    async def add_worker_tasks(self) -> Optional[Iterable[BaseWorker]]:
        if not self.workers:
            return None

        schedule = BackgroundSchedule()
        tasks = []

        for worker_cls, worker_count in self.workers:
            for i in range(worker_count):
                task = worker_cls(name=f'{worker_cls.name_prefix}_{i}', logger=self.logger_ctx_cls(default_logger, {}))
                await task.initialize_worker(app=self)
                schedule.add_task(task)
                tasks.append(task)

        self.on_cleanup.append(schedule.stop)
        await schedule.run(app=self)

        return tasks

    async def add_arbiter_tasks(self, workers: Iterable[BaseWorker]) -> None:
        schedule = BackgroundSchedule()
        task = self.arbiter_cls(name='arbiter', logger=self.logger_ctx_cls(default_logger, {}), workers=workers)
        schedule.add_task(task)

        self.on_cleanup.append(schedule.stop)
        await schedule.run(app=self)

    async def add_stats_tasks(self, workers: Iterable[BaseWorker]) -> None:
        schedule = BackgroundSchedule()
        task = StatsTask(stats_updaters=[worker.update_stats for worker in workers])
        schedule.add_task(task)

        self.on_cleanup.append(schedule.stop)
        await schedule.run(app=self)

    def get_check_handler(self) -> Optional[BaseHandler]:
        return None

    def add_routes(self, *args: Any) -> None:  # type: ignore
        self.router.add_routes(self.routes)

        handler = self.get_check_handler()
        if handler is not None:
            self.router.add_route('GET', r'/check/{verifiable:.+}', handler, name='check')

    def add_sentry(self):
        from sendr_qlog.sentry import sentry_init
        if self.sentry_dsn:
            self.on_cleanup.append(sentry_init(self.sentry_dsn, release=self.version))

    async def prepare_for_tasks(self, app: web.Application) -> None:
        """
        Метод, который вызывается до инициализации всех task'ов.

        Предназначен для переопределения.
        Полезен, например, когда для инициализации воркеров требуется база данных.
        В такой ситуации необходимо создать соединение в prepare_for_tasks.
        """
        pass

    async def setup(self, app: web.Application) -> None:
        self.add_routes()
        self.add_sentry()

        await self.prepare_for_tasks(app)

        workers = await self.add_worker_tasks()
        if workers:
            await self.add_arbiter_tasks(workers)
            await self.add_stats_tasks(workers)

    def start(self, host: str, port: int, loop: Optional[asyncio.AbstractEventLoop] = None) -> None:
        kwargs: Dict[str, Any] = {
            'access_log_format': '%t "%{FOOX-Request-Id}i" "%r" %s %b "%{Referrer}i" "%{User-Agent}i"'
        }
        if not self.debug:
            kwargs['access_log'] = None

        web.run_app(self, host=host, port=port, loop=loop, **kwargs)


class BaseStorageWorkerApplication(BaseWorkerApplication, metaclass=ABCMeta):
    def __init__(
        self,
        db_engine: Optional[EngineUnion] = None,
    ):
        self.db_engine = db_engine
        super().__init__()

    def get_check_handler(self) -> Optional[Type[BaseHandler]]:
        return get_check_handler(self, {'workers-health': WorkersHealthCheckAction}, {'workers': self.workers})

    async def prepare_for_tasks(self, app: web.Application) -> None:
        await super().prepare_for_tasks(app)
        if not self.db_engine:
            self.db_engine = await self.open_engine()

    async def close_engine(self, _: web.Application) -> None:
        if self.db_engine:
            self.db_engine.close()
            await self.db_engine.wait_closed()

    async def setup(self, app: web.Application) -> None:
        await super().setup(app)
        self.on_cleanup.append(self.close_engine)  # type: ignore

    @abstractmethod
    async def open_engine(self) -> EngineUnion:
        """
        Создай и верни экземпляр движка здесь. Обычно это вызов create_configured_engine
        """
        raise NotImplementedError
