import asyncio
from logging import Logger
from typing import Any, ClassVar, List, Optional

from aiohttp import web

from sendr_taskqueue.logger import default_logger


class BackgroundTask:
    pause_period: ClassVar[int] = 1
    app: web.Application

    def __init__(self, *,
                 logger: Optional[Logger] = None,
                 name: Optional[str] = None,
                 once: bool = False,
                 ):
        self.logger = logger or default_logger
        self._name = name
        self._once = once

    @property
    def name(self) -> str:
        return self._name or ''

    async def run(self) -> None:
        try:
            await self._run()
        except asyncio.CancelledError:
            raise
        except Exception as exc:
            self.logger.exception('Exception in task %s', self.name)
            if await self._on_exception(exc):
                raise

    async def pause(self) -> None:
        if self._once:
            raise asyncio.CancelledError
        await asyncio.sleep(self.pause_period)

    async def __call__(self, app: web.Application) -> None:
        self.app = app
        self.logger.info('Starting task %s', self.name)
        try:
            while True:
                await self.run()
                await self.pause()
        except asyncio.CancelledError:
            self.logger.info('Stop task %s by cancel signal', self.name)
            pass
        finally:
            await self._cleanup()
        self.logger.info('Task %s finished', self.name)

    async def _run(self):
        raise NotImplementedError

    async def _on_exception(self, exc: Exception) -> bool:
        pass

    async def _cleanup(self) -> None:
        pass


class BackgroundSchedule:
    def __init__(self, *tasks: BackgroundTask):
        self.tasks: List[BackgroundTask] = []
        self._runs: List[asyncio.Task] = []
        self._waiting_task: Optional[asyncio.Task] = None

        for task in tasks:
            self.add_task(task)

    def add_task(self, task: BackgroundTask) -> None:
        assert self._waiting_task is None, "Can not add task to a running schedule"
        assert isinstance(task, BackgroundTask), "Bad task object for schedule"
        self.tasks.append(task)

    async def run(self, app: web.Application) -> None:
        assert self._waiting_task is None, "Schedule is already running"
        loop = asyncio.get_event_loop()
        self._waiting_task = loop.create_task(self._schedule_tasks_and_wait(app))

    async def stop(self, *args: Any, timeout: int = 10) -> None:
        if self._waiting_task is None:
            return None

        for run in self._runs:
            run.cancel()
        await asyncio.wait_for(self._waiting_task, timeout)

    async def _schedule_tasks_and_wait(self, app: web.Application) -> None:
        loop = asyncio.get_event_loop()
        self._runs = [loop.create_task(x(app)) for x in self.tasks]
        await asyncio.gather(*self._runs, return_exceptions=True)
