import asyncio
import uuid
from asyncio import AbstractEventLoop
from typing import ClassVar, Optional

from aiohttp import web

from sendr_taskqueue.background import BackgroundTask
from sendr_utils import get_hostname


class BaseWorker(BackgroundTask):
    """
    Класс воркера. То, что делает реальную периодическую работу
    Класс должен реализовать:
        Регистрацию
        Отметку о своей работоспособности
        Метод выбора и обработки тасков
        Отметку об окончании работы
    """
    PROCESS_TASK_WITH_NO_PAUSE: ClassVar[bool] = True
    PROCESS_TASK_WITH_PAUSE: ClassVar[bool] = False
    pause_period: ClassVar[int] = 1
    heartbeat_period: ClassVar[int] = 10
    max_retries: ClassVar[int] = 10
    name_prefix: ClassVar[str] = 'task'  # Имя с постфиксом индекса запущенного воркера будет в логах
    no_cancel: ClassVar[bool] = False  # Если таск короткий и его нельзя прерывать
    max_failed_heartbeats: ClassVar[int] = 3  # Количество ошибок отметки активности до остановки воркера

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.worker_id = uuid.uuid4().hex
        self.heartbeat_task = None
        self._host: Optional[str] = None
        self._failed_heartbeats = 0

    @property
    def loop(self) -> AbstractEventLoop:
        return asyncio.get_event_loop()

    @property
    def host(self) -> str:
        if self._host is None:
            self._host = get_hostname()
        return self._host

    @property
    def heartbeat_stopped(self) -> bool:
        return self.heartbeat_task is not None and self.heartbeat_task.done()

    @property
    def active(self) -> bool:
        return self._failed_heartbeats < self.max_failed_heartbeats and not self.heartbeat_stopped

    async def initialize_worker(self, app: web.Application) -> None:
        self.app = app
        await self.register_worker(app)
        self.heartbeat_task = self.loop.create_task(self._heartbeat())

    async def _heartbeat(self) -> None:
        while True:
            try:
                await self.heartbeat()
                self._failed_heartbeats = 0
            except asyncio.CancelledError:
                raise
            except Exception:
                self._failed_heartbeats += 1
                self.logger.exception('Heartbeat error: %d', self._failed_heartbeats)
            await asyncio.sleep(self.heartbeat_period)

    async def _cleanup(self) -> None:
        if self.heartbeat_task:
            self.heartbeat_task.cancel()
        await self.unregister_worker(self.app)

    async def _run(self) -> None:
        need_continue = self.active
        while need_continue:
            if self.no_cancel:
                need_continue = await asyncio.shield(self.process_task())
            else:
                need_continue = await self.process_task()

            if need_continue:
                await asyncio.sleep(0)

    async def update_stats(self, app: web.Application) -> None:
        pass

    async def register_worker(self, app: web.Application) -> None:
        """
        Отметка данных воркера
        """
        raise NotImplementedError

    async def unregister_worker(self, app: web.Application) -> None:
        """
        Отметка окончание работы воркера
        """
        raise NotImplementedError

    async def heartbeat(self) -> None:
        """
        Отметка работоспособности воркера
        """
        raise NotImplementedError

    async def process_task(self) -> bool:
        """
        Выбор и обработка таска (полностью здесь)
        Возврат True означает запустить обработку следующего таска сразу
        """
        raise NotImplementedError
