import os
import signal
from typing import Any, ClassVar, Iterable

from aiohttp import web

from sendr_taskqueue.background import BackgroundTask
from sendr_taskqueue.worker.base import BaseWorker


class BaseArbiterException(Exception):
    pass


class WorkerTaskInactiveArbiterException(BaseArbiterException):
    pass


class BaseArbiterWorker(BackgroundTask):
    """
    Базовый класс арбитра.
    Это периодически запускаемый таск, который обслуживает очередь
    """
    pause_period = 30
    CHECK_WORKERS_ACTIVE: ClassVar[bool] = False
    KILL_ON_CLEANUP: ClassVar[bool] = False
    ARBITER_SIGALRM_ON_CLEANUP_TIMEOUT: ClassVar[int] = 30

    _workers: Iterable[BaseWorker]

    def __init__(self, *args: Any, workers: Iterable[BaseWorker], **kwargs: Any):
        super().__init__(*args, **kwargs)
        self._workers = workers

    async def _run(self) -> None:
        if self.CHECK_WORKERS_ACTIVE:
            await self.check_workers_active()
        await self.clean_tasks(self.app)

    async def _on_exception(self, exc: Exception) -> bool:
        return isinstance(exc, BaseArbiterException)

    async def _cleanup(self) -> None:
        if self.KILL_ON_CLEANUP:
            if self.ARBITER_SIGALRM_ON_CLEANUP_TIMEOUT:
                # Бывает так, что по SIGTERM приложение не завершается.
                # Поэтому считаем, что ARBITER_SIGALRM_ON_CLEANUP_TIMEOUT - graceful период,
                # в течение которого приложение должно завершиться по SIGTERM.
                # А если не получилось, то ядро придёт с SIGALRM и завалит приложение.
                # Так будет, потому что в asyncio и aiohttp не регистрируются обработчики сигнала,
                # а стандартное поведение при получении сигнала в linux - завершение приложения (KILL)
                signal.alarm(self.ARBITER_SIGALRM_ON_CLEANUP_TIMEOUT)
            os.kill(os.getpid(), signal.SIGTERM)

    async def check_workers_active(self) -> None:
        for worker in self._workers:
            if not worker.active:
                raise WorkerTaskInactiveArbiterException(worker.worker_id)

    async def clean_tasks(self, app: web.Application) -> None:
        """
        Всю работу по обслуживанию очереди надо реализовать в переопределенном методе
        """
        pass
