from typing import Dict, List, Optional, Type

from aiohttp import web

from sendr_aiohttp.handler import BaseHandler
from sendr_core import BaseAction
from sendr_taskqueue.worker.base.monitoring.entity import MonitoringStatus, MonitoringVerdict


def get_check_handler(app: web.Application,
                      checkers: Optional[Dict[str, Type[BaseAction]]] = None,
                      params: Optional[dict] = None) -> Type[BaseHandler]:
    checkers = checkers or dict()

    params = params or dict()
    params['app'] = app

    class CheckHandler(BaseHandler):
        async def get(self):
            verifiable = self.request.match_info['verifiable']
            if verifiable not in checkers:
                return web.Response(text='Not found', status=404)

            Checker = checkers[verifiable]
            status: MonitoringStatus = await self.run_action(Checker, params)

            if status.verdict == MonitoringVerdict.OK:
                return web.Response(text='ok')
            else:
                text: List[str] = []
                if status.message:
                    text.append(f'message: "{status.message}"')
                if status.errors:
                    text.append(f'errors: "{", ".join(status.errors)}"')

                code = 400 if status.verdict == MonitoringVerdict.WARN else 500
                return web.Response(text='; '.join(text), status=code)

    return CheckHandler
