from datetime import timedelta
from typing import Any, List, Optional, Tuple, Type

from aiohttp import web

from sendr_core import BaseAction
from sendr_taskqueue.worker.base import BaseWorker
from sendr_taskqueue.worker.base.monitoring.entity import MonitoringStatus, MonitoringVerdict
from sendr_taskqueue.worker.storage import BaseStorageWorker
from sendr_taskqueue.worker.storage.db.entities import WorkerState
from sendr_utils import get_hostname, utcnow


class WorkersHealthCheckAction(BaseAction):
    def __init__(self,
                 app: web.Application,
                 workers: Optional[List[Tuple[Type[BaseStorageWorker], int]]] = None,
                 **kwargs: Any):
        super().__init__()

        self.app = app
        self.workers: Optional[List[Tuple[Type[BaseStorageWorker], int]]] = workers
        self.heartbeat_alert: int = kwargs.get('heartbeat_alert', 180)

    async def handle(self) -> MonitoringStatus:
        result = MonitoringStatus(verdict=MonitoringVerdict.OK)

        if self.workers and len(self.workers) > 0:
            first_worker = self.workers[0][0]

            async with first_worker.storage_context_cls(self.app.db_engine) as storage:  # type: ignore
                worker_mapper = storage[first_worker.mapper_name_worker]

                kwargs = {
                    'state': WorkerState.RUNNING,
                    'host': get_hostname(),
                    'beat_after': utcnow() - timedelta(seconds=self.heartbeat_alert),
                }

                workers = [worker async for worker in worker_mapper.find(**kwargs)]

                if self.workers is not None:
                    tasks: List[Tuple[Type[BaseWorker], int]] = self.workers  # type: ignore
                    for task, amount in tasks:
                        actual_amount = len([worker for worker in workers if task.worker_type == worker.worker_type])
                        if amount != actual_amount:
                            result.verdict = MonitoringVerdict.CRIT
                            result.errors.append(f'{task.worker_type}: {amount} != {actual_amount}')

        return result
