"""
Клиентская сессия aiohttp с добавленными заголовками TVM
"""
import collections.abc
import itertools
from collections.abc import Awaitable
from typing import Any, Callable, ClassVar, Optional, Protocol, Type, TypedDict, Union

import aiohttp
import cachetools
from cachetools.keys import hashkey

from sendr_tvm.common import BaseQTVM

TvmHeaders = TypedDict(
    'TvmHeaders',
    {'X-Ya-Service-Ticket': str},
    total=True,
)


class TvmTicketGetter(Protocol):
    async def get_service_ticket_headers(self, dst: Union[int, str]) -> TvmHeaders:
        raise NotImplementedError


class CachedTvmTicketGetter:
    """
    Кэширующий получатель HTTP заголовков с TVM-тикетами.

    Ленивая инициализацию геттера во время выполнения первого запроса нужна,
    чтобы можно было писать на уровне модуля определение класса сессий с тикет-геттером
    и не запустить случайно какую-нибудь инициализацию под-капотом.
    """

    DEFAULT_TICKET_CACHE_SIZE: ClassVar[int] = 128
    DEFAULT_TICKET_CACHE_TTL: ClassVar[int] = 120

    def __init__(
        self,
        get_tvm: Callable[[], BaseQTVM],
        cache_size: Optional[int] = None,
        cache_ttl: Optional[int] = None,
    ):
        self._ticket_getter = None
        self._ticket_cache: cachetools.TTLCache = cachetools.TTLCache(
            cache_size or self.DEFAULT_TICKET_CACHE_SIZE,
            cache_ttl or self.DEFAULT_TICKET_CACHE_SIZE,
        )
        self._get_tvm = get_tvm

    def _init_ticket_getter(self):
        if self._ticket_getter is None:
            self._ticket_getter = self._get_tvm().ticket_getter()

    async def get_service_ticket_headers(self, dst: Union[int, str]) -> TvmHeaders:
        headers: TvmHeaders
        self._init_ticket_getter()
        key = hashkey(dst)
        try:
            headers = self._ticket_cache[key]
        except KeyError:
            hdrs = self._ticket_getter.get_service_ticket_headers(dst)  # type: ignore
            if isinstance(hdrs, Awaitable):
                hdrs = await hdrs
            headers = hdrs[str(dst)]
            self._ticket_cache[key] = headers
        return headers


class TvmSession(aiohttp.ClientSession):
    """
    Класс сессий с фиксированным TVM source_id.
    Позволяет подписывать исходящие запросы заголовками с TVM-тикетом,
    используя для этого ``CachedTvmTicketGetter``.
    Один экземпляр сессии подразумевается для взаимодействия с одним destination_id.
    """

    tvm_ticket_getter: ClassVar[TvmTicketGetter]

    def __init__(self, tvm_dst, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self._tvm_dst = tvm_dst

    async def _request(self, *args, **kwargs):
        kwargs = await self._fix_headers_kwargs(kwargs)
        return await super()._request(*args, **kwargs)

    async def _ws_connect(self, *args, **kwargs):
        kwargs = await self._fix_headers_kwargs(kwargs)
        return await super()._ws_connect(*args, **kwargs)

    async def _fix_headers_kwargs(self, kwargs: Any) -> Any:
        if self._tvm_dst is None:
            return kwargs
        tvm_headers = await self.tvm_ticket_getter.get_service_ticket_headers(self._tvm_dst)
        headers = kwargs.get('headers', {})
        if isinstance(headers, collections.abc.MutableMapping):
            headers.update(tvm_headers)
        else:
            headers = itertools.chain(headers, tvm_headers.items())
        kwargs['headers'] = headers
        return kwargs


def sessions_producer(
    get_tvm: Callable[[], BaseQTVM],
    cache_size: Optional[int] = None,
    cache_ttl: Optional[int] = None,
) -> Type[TvmSession]:
    """
    Возвращает класс сессий, умеющих подписывать исходящие запросы TVM тикетами
    Предполагается, что сессия будет относительно короткоживущая.

    :param get_tvm:
        callable, возвращающий TVM интерфейс,
        через который можно получать тикеты для одного конкретного TVM-источника.

        Принимаем callable, так как потенциально реализации TVM интерфейса бывают stateful,
        поэтому логика создания TVM должна быть на уровне старта приложения,
        так как даже для TVM интерфейса, реализованный через обращение к HTTP-демону на localhost,
        можно захотеть сделать Keep-Alive и свой TCP Connector.

        Позднее связывание сессии с получателем тикетов также позволяет проще писать тесты:
        можно мокнуть хост и порт tvm клиента, даже если класс клиентских сессий приложения
        уже создан на уровне модуля.

    :param cache_size: макисмальный размер кеша под тикеты
    :param cache_ttl: максимальное время хранения тикета в кеше
    """

    class _TvmSession(TvmSession):
        tvm_ticket_getter = CachedTvmTicketGetter(get_tvm, cache_size, cache_ttl)

    return _TvmSession
