import os
from typing import Union

from .exceptions import TicketParsingException, TvmException, TVMNetworkError, TVMResponseError  # NOQA


class BaseQTVM:
    """
    Класс для проверки тикетов в Qloud.
    Конфигурация берется из настроек окружения Qloud
    """

    QTICKET_GETTER = None
    QTICKET_CHECKER = None

    def __init__(self, client_name: str, host: str = 'localhost', port: int = 1):
        self._client_name = client_name
        self.host = host
        self.port = port

        self._inited = False

    @property
    def q_token(self) -> str:
        return os.environ.get('QLOUD_TVM_TOKEN', '') or os.environ.get('TVMTOOL_LOCAL_AUTHTOKEN', '')

    def _init_clients(self) -> None:
        if self._inited:
            return

        if not self.q_token:
            raise TvmException

        if not self.client_name:
            raise TvmException

        self._inited = True

    @property
    def client_name(self) -> str:
        return self._client_name

    @property
    def url(self):
        return f'http://{self.host}:{self.port}'

    def ticket_getter(self):
        self._init_clients()
        if not self.QTICKET_GETTER:
            raise TvmException
        return self.QTICKET_GETTER(self)

    def ticket_checker(self):
        self._init_clients()
        if not self.QTICKET_CHECKER:
            raise TvmException
        return self.QTICKET_CHECKER(self)


class TicketCheckResult:
    """
    Результат проверки тикетов из заголовков.
    TVM2 идентифицируется сервисным тикетом.
    """

    def __init__(self, service, user):
        self._service_ticket = service
        self._user_ticket = user

    @property
    def valid(self):
        """ У нас есть хотя бы один сервисный тикет """
        return self._service_ticket is not None

    @property
    def src(self):
        """ От кого пришел тикет """
        if self._service_ticket:
            return self._service_ticket.get('src')
        return None

    @property
    def scopes(self):
        if self._service_ticket:
            return self._service_ticket.get('scopes')
        return ()

    @property
    def uids(self):
        """ Набор uid'ов из пользовательского токена. Для легаси - только safe_uids"""
        if self._service_ticket and self._user_ticket:
            return self._user_ticket.get('uids', ())
        return ()

    @property
    def default_uid(self):
        """ uid по-умолчанию """
        if self._service_ticket and self._user_ticket:
            return self._user_ticket.get('default_uid')
        return None

    def has_src(self, src):
        src = int(src)
        if self._service_ticket:
            return self._service_ticket.get('src') == src
        return False

    def has_scope(self, scope):
        return False

    def has_uid(self, uid):
        if self._service_ticket and self._user_ticket:
            return uid in self._user_ticket.get('uids', ())
        return False


class BaseQTicketGetter:
    """
    Класс для получения тикетов через API Qloud
    """
    _tvm: BaseQTVM

    def __init__(self, tvm_interface: BaseQTVM):
        self._tvm = tvm_interface

    @staticmethod
    def _fix_list(value):
        """ Если нужен именно list """
        if isinstance(value, list):
            return value
        elif isinstance(value, (set, tuple)):
            return list(value)
        else:
            return [value]

    def _tvm_tickets_request_kwargs(self, destinations):
        return {
            'url': f'{self._tvm.url}/tvm/tickets',
            'params': {
                'dsts': ','.join(map(str, self._fix_list(destinations))),
                'src': self._tvm.client_name,
            },
            'headers': {
                'Authorization': self._tvm.q_token,
            },
        }

    def _get_headers_by_destinations(self, resp_data):
        headers_by_destinations = {}

        for k, v in resp_data.items():
            headers_by_destinations[str(v['tvm_id'])] = {
                'X-Ya-Service-Ticket': v['ticket']
            }
            headers_by_destinations[k] = {
                'X-Ya-Service-Ticket': v['ticket']
            }

        return headers_by_destinations

    # Методы

    def get_service_ticket_headers(self, destinations):
        raise NotImplementedError

    def get_service_ticket(self, destination):
        raise NotImplementedError


def get_tvm_checksrv_kwargs(tvm_url: str, tvm_id: Union[str, int], tvm_token: str, tvm_ticket: str) -> dict:
    return {
        'url': f'{tvm_url}/tvm/checksrv',
        'params': {
            'dst': tvm_id,
        },
        'headers': {
            'Authorization': tvm_token,
            'X-Ya-Service-Ticket': tvm_ticket,
        },
    }


def get_tvm_checkusr_kwargs(tvm_url: str, tvm_token: str, tvm_ticket: str) -> dict:
    return {
        'url': f'{tvm_url}/tvm/checkusr',
        'headers': {
            'Authorization': tvm_token,
            'X-Ya-User-Ticket': tvm_ticket,
        },
    }


class BaseQTickerChecker:
    _tvm: BaseQTVM
    HEADER_SERVICE_TICKET = 'x-ya-service-ticket'
    HEADER_USER_TICKET = 'x-ya-user-ticket'

    def __init__(self, tvm_interface: BaseQTVM):
        self._tvm = tvm_interface

    def _get_tvm_headers(self, headers):
        return {
            k.lower(): v
            for k, v in headers.items()
            if k.lower() in (self.HEADER_SERVICE_TICKET, self.HEADER_USER_TICKET)
        }

    def _tvm_checksrv_request_kwargs(self, ticket: str) -> dict:
        return get_tvm_checksrv_kwargs(self._tvm.url, self._tvm.client_name, self._tvm.q_token, ticket)

    def _tvm_checkusr_request_kwargs(self, ticket):
        return get_tvm_checkusr_kwargs(self._tvm.url, self._tvm.q_token, ticket)

    async def check_headers(self, headers):
        raise NotImplementedError
