from dataclasses import dataclass
from typing import Union

import aiohttp

from sendr_tvm.common import (
    BaseQTickerChecker, BaseQTicketGetter, BaseQTVM, TicketCheckResult, get_tvm_checksrv_kwargs, get_tvm_checkusr_kwargs
)

from .common.exceptions import TicketParsingException, TvmException, TVMNetworkError, TVMResponseError  # NOQA


class _NetTvmAsync:
    async def _make_get(self, **kwargs):
        try:
            async with aiohttp.ClientSession() as session:
                request = session.get(**kwargs)
                async with request as response:
                    if response.status == 403:
                        error = await response.json()
                        raise TicketParsingException(
                            message=error.get('error'),
                            status=403,
                            debug_info=error.get('debug_string'),
                        )
                    response.raise_for_status()
                    return await response.json()
        except aiohttp.ClientConnectionError:
            raise TVMNetworkError
        except aiohttp.ServerTimeoutError:
            raise TVMResponseError
        except aiohttp.ClientError:
            raise TVMResponseError


class QTicketGetter(BaseQTicketGetter, _NetTvmAsync):
    async def get_service_ticket_headers(self, destinations):
        resp_data = await self._make_get(**self._tvm_tickets_request_kwargs(destinations))
        return self._get_headers_by_destinations(resp_data)

    async def get_service_ticket(self, destination):
        resp_data = await self._make_get(**self._tvm_tickets_request_kwargs(destination))
        for k, v in resp_data.items():
            return v['ticket']


class QTickerChecker(BaseQTickerChecker, _NetTvmAsync):
    async def check_headers(self, headers):
        tvm_headers = self._get_tvm_headers(headers)

        service_ticket = None
        user_ticket = None

        if self.HEADER_SERVICE_TICKET in tvm_headers:
            header = tvm_headers[self.HEADER_SERVICE_TICKET]
            service_ticket = await self._make_get(**self._tvm_checksrv_request_kwargs(header))

        if self.HEADER_USER_TICKET in tvm_headers:
            header = tvm_headers[self.HEADER_USER_TICKET]
            user_ticket = await self._make_get(**self._tvm_checkusr_request_kwargs(header))

        return TicketCheckResult(service_ticket, user_ticket)


class QTVM(BaseQTVM):
    QTICKET_GETTER = QTicketGetter
    QTICKET_CHECKER = QTickerChecker


# В некотором смысле дублирующая, но оторванная и без множественного наследования реализация получения service_ticket

@dataclass
class ServiceTicket:
    src: int


@dataclass
class UserTicket:
    uid: int


class TvmChecker(_NetTvmAsync):
    def __init__(
        self,
        tvm_src: Union[int, str],
        tvm_url: str,
        tvm_token: str,
    ):
        self.tvm_src = tvm_src
        self.tvm_token = tvm_token
        self.tvm_url = tvm_url

    async def parse_service_ticket(self, tvm_ticket: str) -> ServiceTicket:
        service_ticket_response = await self._make_get(
            **get_tvm_checksrv_kwargs(self.tvm_url, self.tvm_src, self.tvm_token, tvm_ticket)
        )

        return ServiceTicket(service_ticket_response['src'])

    async def parse_user_ticket(self, tvm_ticket: str) -> UserTicket:
        response = await self._make_get(
            **get_tvm_checkusr_kwargs(self.tvm_url, self.tvm_token, tvm_ticket)
        )

        return UserTicket(response['default_uid'])
