import json
import urllib.request
from urllib.error import HTTPError, URLError
from urllib.parse import urlencode

from sendr_tvm.common import BaseQTickerChecker, BaseQTicketGetter, BaseQTVM, TicketCheckResult

from .common.exceptions import TicketParsingException, TVMNetworkError, TVMResponseError


class _NetTvmSync:
    def _make_get(self, **kwargs):
        url = kwargs.pop('url', None)
        params = kwargs.pop('params', None)
        if params:
            url = f'{url}?{urlencode(params)}'

        try:
            req = urllib.request.Request(url=url, **kwargs)
            resp = urllib.request.urlopen(req)
            resp_data = resp.read().decode('utf-8')
            resp.close()

            if resp.getcode() == 403:
                error = json.loads(resp_data)
                raise TicketParsingException(
                    message=error.get('error'),
                    status=403,
                    debug_info=error.get('debug_string'),
                )
            elif resp.getcode() >= 400:
                raise TicketParsingException(
                    message="Unknown exception",
                    status=resp.getcode(),
                    debug_info=resp_data
                )
            else:
                return json.loads(resp_data)
        except HTTPError as e:
            if e.code == 403:
                error = json.loads(e.read().decode('utf-8'))
                raise TicketParsingException(
                    message=error.get('error'),
                    status=403,
                    debug_info=error.get('debug_string'),
                )
            raise e
        except URLError as e:
            if isinstance(e.reason, ConnectionRefusedError):
                raise TVMNetworkError
            raise TVMResponseError


class QTicketGetter(BaseQTicketGetter, _NetTvmSync):
    def get_service_ticket_headers(self, destinations):
        resp_data = self._make_get(**self._tvm_tickets_request_kwargs(destinations))
        return self._get_headers_by_destinations(resp_data)

    def get_service_ticket(self, destination):
        resp_data = self._make_get(**self._tvm_tickets_request_kwargs(destination))
        for k, v in resp_data.items():
            return v['ticket']


class QTickerChecker(BaseQTickerChecker, _NetTvmSync):
    def check_headers(self, headers):
        tvm_headers = self._get_tvm_headers(headers)

        service_ticket = None
        user_ticket = None

        if self.HEADER_SERVICE_TICKET in tvm_headers:
            header = tvm_headers[self.HEADER_SERVICE_TICKET]
            service_ticket = self._make_get(**self._tvm_checksrv_request_kwargs(header))

        if self.HEADER_USER_TICKET in tvm_headers:
            header = tvm_headers[self.HEADER_USER_TICKET]
            user_ticket = self._make_get(**self._tvm_checkusr_request_kwargs(header))

        return TicketCheckResult(service_ticket, user_ticket)


class QTVM(BaseQTVM):
    QTICKET_GETTER = QTicketGetter
    QTICKET_CHECKER = QTickerChecker
