from functools import wraps
from typing import Type, TypeVar

_ABSTRACT_CLASS_MARK = '_qtools_is_abstract'

T = TypeVar('T')


def _process_abstract_class(cls: Type[T]) -> Type[T]:
    setattr(cls, _ABSTRACT_CLASS_MARK, True)

    init = cls.__dict__.get('__init__')

    def init_check_abstract(self, *args, **kwargs):
        if type(self) is cls:
            raise RuntimeError(f'Class {cls.__name__} is an abstract class and cannot be instantiated')
        if init:
            init(self, *args, **kwargs)
        else:
            super(cls, self).__init__(*args, **kwargs)

    if init:
        init_check_abstract = wraps(init)(init_check_abstract)
    setattr(cls, '__init__', init_check_abstract)

    return cls


def abstractclass(cls: Type[T]) -> Type[T]:
    """
    Restricts direct instantiation of a given class.
    """

    def wrap(cls):
        return _process_abstract_class(cls)

    if cls is None:
        # Caller called @abstractclass
        return wrap

    # Caller called @abstractclass()
    return wrap(cls)


def isabstract(cls: Type[T]) -> bool:
    return _ABSTRACT_CLASS_MARK in cls.__dict__
