import json
import subprocess
from functools import lru_cache
from os import path
from typing import List

from jinja2 import FileSystemLoader as JinjaFileSystemLoader
from jinja2 import TemplateNotFound

try:
    from jinja2.loaders import split_template_path
    from library.python import resource

    class FileSystemLoader(JinjaFileSystemLoader):
        """
        Переопределяет методы JinjaFileSystemLoader
        для чтения шаблонов не с диска, а из ресурсов
        """

        def get_source(self, environment, template):
            pieces = split_template_path(template)
            for searchpath in self.searchpath:
                filename = path.join(searchpath, *pieces)
                content = resource.find(filename)
                if content is None:
                    continue
                return content.decode(self.encoding), filename, lambda: True
            raise TemplateNotFound(template)

        def list_templates(self):
            found = set()
            for searchpath in self.searchpath:
                filenames = resource.iterkeys(searchpath)
                for filename in filenames:
                    template = filename[len(searchpath):] \
                        .strip(path.sep) \
                        .replace(path.sep, '/')
                    if template[:2] == './':
                        template = template[2:]
                    found.add(template)
            return sorted(found)
except ImportError:
    FileSystemLoader = JinjaFileSystemLoader  # type: ignore


class Font:
    """
    Функция шабалонизатора, которая позволяет по названию шрифта и стилю
    получить css-блок @font-face c указанием абсолютного пути до файла со шрифтом.
    Необходимо для корректного рендеринга кириллицы
    Пример: env.globals.update({'font': Font()})
    В шаблоне: {{font('Times New Roman', 'Regular')}}
    """

    @lru_cache(None)
    def font_list(self) -> List[dict]:
        cmd = r'fc-list --format=\{\"names\":\"%{family}\",\"path\":\"%{file}\",\"styles\":\"%{style}\"\}\n'
        process = subprocess.Popen(cmd.split(), stdout=subprocess.PIPE)
        output, error = process.communicate()

        if error:
            raise RuntimeError(f'Error during run fc-list: {str(error)}')

        result = []
        for line in output.decode('utf-8').strip().split('\n'):
            font = json.loads(line)

            font['names'] = list(map(str.lower, font['names'].split(',')))
            font['styles'] = list(map(str.lower, font['styles'].split(',')))
            font['format'] = font['path'].split('.', 2)[-1].lower()

            result.append(font)
        return result

    def __call__(self, fontname: str, style: str) -> str:
        assert fontname and style, 'Fontname and style can\'t be empty'

        font = next(
            filter(
                lambda x: fontname.lower() in x['names'] and style.lower() in x['styles'],
                self.font_list()
            ),
            None
        )

        return f'''
            @font-face {{
             font-family: "{fontname}";
             src: url("{font['path']}") format("{font['format']}");
            }}
        ''' if font else ''
