from asyncio import sleep
from functools import wraps
from itertools import chain
from typing import Callable, Tuple, Type


class RetryLimitExceededError(Exception):
    def __init__(self, last_exception: Exception):
        self.last_exception: Exception = last_exception


def async_retry(base_delay: float,
                retries: int,
                exceptions: Tuple[Type[Exception]],
                wrap_exception: bool = True) -> Callable:
    """
    Декоратор, обеспечивающий экспоненциальные ретраи, если в асинхронной функции происходят
    заданные исключения.

    wrap_exception - нужно ли оборачивать последнее исключение в RetryLimitExceeded
    """
    def _inner(f):
        @wraps(f)
        async def _retry(*args, **kwargs):
            last_exception = None

            for delay in chain((0,), (base_delay * 2**i for i in range(retries))):
                if delay > 0:
                    await sleep(delay)
                try:
                    return await f(*args, **kwargs)
                except exceptions as ex:
                    last_exception = ex

            if wrap_exception:
                raise RetryLimitExceededError(last_exception)

            raise last_exception

        return _retry
    return _inner
