from typing import Any, Callable, Dict, Generic, Iterable, List, TypeVar, cast, get_args, get_origin

import marshmallow as mm

from sendr_utils.schemas.base import BaseSchema

DataClassType = TypeVar('DataClassType')


class UnmarshalOneResult(mm.UnmarshalResult, Generic[DataClassType]):
    """
    Аннотация поля errors в строгом режиме использования схем не нужна,
    так как воспользоваться результатом десериализации получится только если не было ошибок валидации:
    при возникновении ошибок валидации будет брошено исключение.
    """
    data: DataClassType
    errors: Any


class UnmarshalManyResult(mm.UnmarshalResult, Generic[DataClassType]):
    data: List[DataClassType]
    errors: Any


class DataClassLoadingSchemaMeta(mm.schema.SchemaMeta):
    """
    Внедряем post_load(many=False) хук.
    Проверяем, что он зарегистрирован в классе схемы, и других подобных хуков нет.
    """

    def __new__(mcls, name, bases, attrs):
        """Внедряем хук."""
        attrs['_call_create_data_class_instance'] = _call_create_data_class_instance
        cls = super().__new__(mcls, name, bases, attrs)
        return cls

    def __init__(cls, *args, **kwargs):
        """Проверяем консистентность хуков в созданном классе."""
        super().__init__(*args, **kwargs)
        cls._ensure_create_dataclass_hook_is_one()

    def _ensure_create_dataclass_hook_is_one(cls):
        # немного грязи и знаний внутреннего устройства marshmallow 2
        hooks: List[str] = cls.__processors__[(mm.decorators.POST_LOAD, False)]

        if not (
            len(hooks) == 1
            and getattr(cls, hooks[0], None) is _call_create_data_class_instance
        ):
            # принуждение иногда полезно?
            raise RuntimeError(
                f'Please do not use or change post_load(many=False) hooks for {cls}. '
                f'Use custom create_data_class_instance method.'
            )


@mm.post_load(pass_many=False, pass_original=False)
def _call_create_data_class_instance(self: DataClassLoadingSchemaMeta, loaded_data: Dict[str, Any]) -> DataClassType:
    """
    Использование нескольких post_load(pass_many=False) не рекомендуется, так как нет
    проработанного механизма соблюдения порядка вызова нескольких хуков, вместо этого
    предлагается в производном классе переопределить метод create_data_class_instance.
    """
    return self.create_data_class_instance(loaded_data)


class BaseDataClassSchema(BaseSchema, Generic[DataClassType], metaclass=DataClassLoadingSchemaMeta):
    """
    Оборачивает результат десериализации в датакласс.

    Учти, что хуки типа @validates_schema увидят в общем случае промежуточное состояние
    данных: вложенные схемы к этому моменту отработают и в аргументах будет
    словарь, значениями которого будут датаклассы.
    """

    # аннотация Callable дана для универсальности:
    # всякий класс является Callable, возвращающим значение своего типа.
    @property
    def data_class(self) -> Callable[..., DataClassType]:
        """
        Возвращает класс, в который надо "заэнвелопить" результат load'инга схемы.
        Реализация "по-умолчанию" пытается взять класс из аргументов дженерика.

        Без этого хака придётся везде бойлерплейтить:
        class MyDataclassSchema(BaseDataClassSchema[MyDataclass]):
            @property
            def data_class(self):
                return MyDataclass

        Можно было бы и проще:
        class MyDataclassSchema(BaseDataClassSchema[MyDataclass]):
            data_class = MyDataclass

        Но это ломает систему типов. Поэтому никак.
        """
        return self._get_dataclass_from_generic_args(generic_origin=BaseDataClassSchema)

    def _get_dataclass_from_generic_args(self, generic_origin: type) -> Callable[..., DataClassType]:
        """Достаёт аргумент джеенерика и возвращает его.

        PEP https://www.python.org/dev/peps/pep-0560/
        Дока https://docs.python.org/3/library/typing.html#typing.get_args.
        Если отнаследоваться от специфицированного генерика и сделать этого наследника генериком,
        то аргумент корневого генерика (как и сам генерик) будет сокрыт. В этом заключается недостаток
        такой реализации"""
        bases = list(type(self).__orig_bases__)
        for base in bases:
            if get_origin(base) is generic_origin:
                return get_args(base)[0]
        raise RuntimeError('Base not found')

    # public API for usage or extension
    def create_data_class_instance(self, loaded_data: Dict[str, Any]) -> DataClassType:
        """
        Создает экземпляр датакласса.
        В производных схемах предлагается переопределять
        данный метод вместо использования post_load(many=False) хуков.
        """
        return self.data_class(**loaded_data)

    def load_one(self, data: Dict[str, Any]) -> DataClassType:
        result, _ = self.load(data, many=False)
        return cast(DataClassType, result)

    def load_many(self, data: Iterable[Dict[str, Any]]) -> List[DataClassType]:
        result, _ = self.load(data, many=True)
        return cast(List[DataClassType], result)
