from collections import defaultdict
from decimal import Decimal
from typing import Any, Collection, Dict, Optional, cast

from marshmallow import exceptions, validate


class NonEmptyString(validate.Validator):
    error_none = 'Field may not be null.'
    error_invalid_type = 'Invalid type.'
    error = 'String should not be empty.'

    def __call__(self, value: str) -> str:
        if value is None:
            raise exceptions.ValidationError(self.error_none)

        if not isinstance(value, str):
            raise exceptions.ValidationError(self.error_invalid_type)

        if len(value.strip()) == 0:
            raise exceptions.ValidationError(self.error)

        return value


class DecimalExponentValidator(validate.Validator):
    error_invalid_type = 'Invalid type: Decimal expected.'
    error = 'Invalid exponent: expected {expected}, got {actual}.'

    def __init__(self, exponent: int = 2):
        self.exponent = exponent

    def _repr_args(self) -> str:
        return "exponent={!r}".format(self.exponent)

    def __call__(self, value: Any) -> Decimal:
        if not isinstance(value, Decimal):
            raise exceptions.ValidationError(self.error_invalid_type)

        expected = abs(self.exponent)
        actual = abs(value.as_tuple().exponent)

        if actual > expected:
            error = self.error.format(expected=expected, actual=actual)
            raise exceptions.ValidationError(error)

        return value


class UniqueValues(validate.Validator):
    error_invalid_type = 'Invalid type: Iterable expected.'
    error = 'Duplicate {name} with value {attr}'

    def __init__(self, attribute: Optional[str] = None):
        self.attribute = attribute

    def __call__(self, value: Collection[Any]) -> Any:
        attr2count: Dict[Any, int] = defaultdict(int)
        try:
            for el in value:
                attr = getattr(el, self.attribute) if self.attribute else el
                attr2count[attr] += 1
                if attr2count[attr] > 1:
                    raise exceptions.ValidationError(
                        self.error.format(
                            name=self.attribute if self.attribute else 'item',
                            attr=attr,
                        )
                    )
        except TypeError:
            raise exceptions.ValidationError(self.error_invalid_type)

        return value


class And(validate.Validator):
    """
    Стырено из 3й версии Marshmallow
    https://a.yandex-team.ru/arc/trunk/arcadia/contrib/python/marshmallow/py3/marshmallow/validate.py?rev=r8648900#L43
    """

    default_error_message = "Invalid value."

    def __init__(self, *validators: validate.Validator, error: Optional[str] = None):
        self.validators = tuple(validators)
        self.error = error or self.default_error_message  # type: str

    def _repr_args(self) -> str:
        return "validators={!r}".format(self.validators)

    def __call__(self, value: Any) -> Any:
        errors = []
        kwargs = {}
        for validator in self.validators:
            try:
                r = validator(value)
                if not isinstance(validator, validate.Validator) and r is False:
                    raise exceptions.ValidationError(self.error)
            except exceptions.ValidationError as err:
                kwargs.update(err.kwargs)
                if isinstance(err.messages, dict):
                    errors.append(err.messages)
                else:
                    # FIXME : Get rid of cast
                    errors.extend(cast(list, err.messages))
        if errors:
            raise exceptions.ValidationError(errors, **kwargs)
        return value
