import asyncio
import contextlib
import os
from dataclasses import asdict, dataclass
from enum import Enum
from pathlib import Path
from typing import Any, ClassVar, Dict, Optional, Union

import ujson
from multidict import CIMultiDict, CIMultiDictProxy

import sendr_qstats
from sendr_utils import utcnow
from sendr_writers.base.writer import Writer

CLOSE_TIMEOUT = 5


@dataclass
class BaseLog:
    MESSAGE: ClassVar[str]

    def __post_init__(self):
        self.timestamp = utcnow()

    def dump(self):
        data = asdict(self)
        for key, value in data.items():
            if isinstance(value, Enum):
                data[key] = value.value
        data['message'] = self.MESSAGE
        data['timestamp'] = self.timestamp.strftime("%Y-%m-%d %H:%M:%S")
        return data


@dataclass
class InteractionResponseLog(BaseLog):
    MESSAGE = 'INTERACTION_RESPONSE'

    request_id: str
    request_url: str
    request_method: str
    request_kwargs: Dict[str, Any]
    response: Any
    response_headers: Union[CIMultiDictProxy[str], CIMultiDict[str]]
    status: int
    exception_type: Optional[str] = None
    exception_message: Optional[str] = None


class Pusher:
    """
    Пушеры для логирования в файл. Не рекомендуются к использованию, нуждаются в доработке.
    Имеют следующие проблемы: unsafe multiprocessing, зависание на ротации, ротация по времени, а не объему
    """
    def __init__(
        self,
        log_path: str,
        rotate_interval: int,
        backup_count: int,
        loop: Optional[asyncio.AbstractEventLoop] = None,
        push_time_metric: Optional[sendr_qstats.Histogram] = None,
        compress_on_backup: bool = False,
    ):
        self.loop = loop or asyncio.get_event_loop()
        self._init_path(log_path)
        self.writer = Writer(
            base_file_name=log_path,
            rotate_interval=rotate_interval,
            backup_count=backup_count,
            loop=self.loop,
            compress_on_backup=compress_on_backup,
        )
        self.writer.run()
        self.push_time_metric = push_time_metric

    @staticmethod
    def _init_path(log_path: str) -> None:
        try:
            os.makedirs(os.path.dirname(log_path))
            Path(log_path).touch()
        except (FileNotFoundError, FileExistsError):
            # Предполагаем что при этих ошибках уже все пути есть
            pass

    async def _push(self, data: dict) -> None:
        data_dump = ujson.dumps(data, ensure_ascii=False, escape_forward_slashes=False)
        await self.writer.write(data_dump.encode('utf-8'))

    async def push(self, log: BaseLog) -> None:
        with contextlib.ExitStack() as ctx:
            if self.push_time_metric is not None:
                ctx.enter_context(self.push_time_metric.time)

            await self._push(log.dump())

    async def close(self):
        await self.writer.close()


class CommonPushers:
    response_log: Optional[Pusher] = None
