import asyncio
import gzip
import os
import shutil
from asyncio import AbstractEventLoop, Task
from datetime import date, datetime, time
from typing import Optional, Tuple

import aiofiles
from aiofiles.threadpool import AsyncFileIO

CLOSE_TIMEOUT = 5


class WriterError(Exception):
    pass


class WriterClosed(WriterError):
    pass


def _move_file(source: str, dest: str) -> None:
    if not os.path.exists(source):
        return
    shutil.move(source, dest)


def _compress_file(source: str, dest: str) -> None:
    with open(source, 'rb') as f_in, gzip.open(dest, 'wb') as f_out:
        shutil.copyfileobj(f_in, f_out)


class Writer:
    def __init__(
        self,
        *,
        base_file_name: str,
        rotate_interval: int,
        backup_count: int,
        loop: AbstractEventLoop,
        queue_size: int = 1024,
        compress_on_backup: bool = False,
    ):
        self.loop = loop
        self.base_file_name = base_file_name
        self.rotate_interval = rotate_interval
        self.backup_count = backup_count

        self.file: Optional[AsyncFileIO] = None

        self._queue: asyncio.Queue = asyncio.Queue(queue_size)
        self._running = False
        self._write_task: Optional[Task] = None
        self._tick_task: Optional[Task] = None
        self._cur_rotate_period = self._get_rotating_period()

        self.compress_on_backup = compress_on_backup
        self._suffix = '.gz' if self.compress_on_backup else ''

    @property
    def running(self) -> bool:
        return self._running

    async def write(self, record: bytes) -> None:
        if not self._running:
            raise WriterClosed
        await self._queue.put(record)

    def run(self) -> None:
        if self._running:
            return
        # Запускаем 2 таски:
        #  -одна будет разбирать очередь записей и отправлять их в файл
        #  -вторая раз в минуту будет отправлять пустое сообщение в очередь
        #   чтобы инициировать ротацию файла
        self._write_task = self.loop.create_task(self._writer())
        self._tick_task = self.loop.create_task(self._ticker())
        self._running = True

    async def close(self) -> None:
        if not self._running:
            raise WriterClosed
        self._running = False
        if self._tick_task:
            self._tick_task.cancel()
        if self._write_task:
            try:
                await asyncio.wait_for(self._close_task(), timeout=CLOSE_TIMEOUT)
            except asyncio.TimeoutError:
                self._write_task.cancel()
                await self._close()

    async def _writer(self) -> None:
        while True:
            record = await self._queue.get()
            if record is ...:
                self._queue.task_done()
                await self._close()
                return
            await self._write_record(record)
            self._queue.task_done()

    async def _ticker(self) -> None:
        while True:
            await asyncio.sleep(60)
            await self._queue.put(None)

    async def _write_record(self, record: bytes) -> None:
        await self._rotate()
        await self._open()

        # Пустая запись предназначена для ротации файла
        if record is None:
            return

        if self.file is None:  # make mypy happy
            raise RuntimeError('File was not opened')

        await self.file.write(record)
        await self.file.write(b'\n')
        await self.file.flush()

    async def _rotate(self) -> None:
        if not self.file:
            return

        # Проверяем не перешли ли еще в новый "период ротации"
        new_rotate_period = self._get_rotating_period()
        if new_rotate_period == self._cur_rotate_period:
            return

        # Если открыт пустой файл, то не ротируем
        if await self.file.tell() == 0:
            return

        self._cur_rotate_period = new_rotate_period
        await self._close()
        if self.backup_count > 0:
            file_name = self.base_file_name + self._suffix
            for i in range(self.backup_count - 1, 0, -1):
                sfn = f'{file_name}.{i}'
                dfn = f'{file_name}.{i + 1}'
                _move_file(sfn, dfn)

            dfn = f'{file_name}.1'
            if self.compress_on_backup:
                # compress the current log file on rotation
                await self.loop.run_in_executor(
                    None, _compress_file, self.base_file_name, dfn
                )
                os.remove(self.base_file_name)
            else:
                _move_file(self.base_file_name, dfn)

    async def _open(self) -> AsyncFileIO:
        if self.file is None:
            self.file = await aiofiles.open(self.base_file_name, 'ab')
        return self.file

    async def _close(self) -> None:
        if not self.file:
            return
        try:
            await self.file.flush()
        finally:
            await self.file.close()
        self.file = None

    async def _close_task(self) -> None:
        await self._queue.put(...)
        if self._write_task:
            await self._write_task

    def _get_rotating_period(self) -> Tuple[date, int]:
        """
        Считаем "период" для ротации файлов. Он состоит из даты и интервала внутри дня.
        """
        now = datetime.now()
        now_date = now.date()
        day_time = int((now - datetime.combine(now_date, time())).total_seconds())
        period_in_date = day_time // self.rotate_interval
        return now_date, period_in_date
