import logging

from asyncpgsa.log import query_logger

from mail.python.theatre.app.env import EnvType
from mail.python.theatre.app.settings.log import LogSettings

from .async_logs import file_handler
from .json_logs import monkey_patch_logger
from .postgre_conn import ADDITIONAL_LOG_LEVELS


def init_logging(access_log_path=None, service_log_path=None, error_log_path=None, log_level='info', custom_loggers=None):
    for level_name, level in ADDITIONAL_LOG_LEVELS.items():
        logging.addLevelName(level, level_name)

    monkey_patch_logger()
    logging.getLogger().setLevel(logging.getLevelName(log_level.upper()))

    if access_log_path:
        access_log.addHandler(file_handler(access_log_path))
    access_log.propagate = False

    if error_log_path:
        error_handler = file_handler(error_log_path, level=logging.ERROR)
        logging.getLogger().addHandler(error_handler)

    if service_log_path:
        logging.getLogger().addHandler(file_handler(service_log_path))
    else:
        logging.getLogger().addHandler(logging.StreamHandler())
    # Effectively disable logging of query compilations
    query_logger.setLevel(logging.CRITICAL)

    if custom_loggers:
        for logger, file in custom_loggers:
            logger.addHandler(file_handler(file))


def init_app_logging(settings: LogSettings, env: EnvType):
    settings.dir_path.mkdir(mode=0o755, exist_ok=True, parents=True)
    init_logging(
        access_log_path=settings.dir_path / 'access.log',
        service_log_path=settings.dir_path / 'service.log',
        error_log_path=settings.dir_path / 'error.log',
        log_level=settings.level,
        custom_loggers=[(log.get_logger(), settings.dir_path / log.file) for log in settings.custom_loggers],
    )


access_log = logging.getLogger('aiohttp.access')
